----------------------------------------------------------------------------------
-- Company: The University of Oklahoma
-- Engineer: Arne Schwettmann
-- 
-- Create Date:    18:01:58 03/01/2010 
-- Design Name: 
-- Module Name:    highPass16bit - Behavioral 
-- Project Name:   Shaffer Lab Lockbox
-- Target Devices: Digilent Nexys2 1200k Spartan3 based evaluation board
-- Tool versions: 
-- Description: 
--    please see the top level source file "lockbox.vhdl" for release info 
--
-- Dependencies: 
--
-- Revision: 
--    1.0 public release (August 2011)
--
-- Additional Comments: 
--
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

--attribute mult_style: string; 

--attribute mult_style of pidPiezo: pidPiezo is "block";

entity pidPiezo is
   port (
      clk            : in std_logic;
      input     : in signed(15 downto 0);
      output   : out signed(15 downto 0);
      setPoint : in signed(15 downto 0);
      initOutput : in signed(15 downto 0);
      Kp        : in unsigned(23 downto 0);
      Ki        : in unsigned(23 downto 0);
      Kd        : in unsigned(23 downto 0);
      gain     : in unsigned(23 downto 0);
      bypass   : in std_logic;
      reset    : in std_logic
   );
end pidPiezo;

architecture Behavioral of pidPiezo is

   signal currentOutput       : signed(35 downto 0);
  
begin

   -- implementation of discrete time PID filter TYPE C as described in 
   -- http://bestune.50megs.com/typeABC.htm
   -- Ts is the effective clock period, so it is 50 MHz/13 and 1/TS is a shift to the left
   -- by ~22 bits
   -- shift it by less (19 bits) to make it easier to control from the interface

   -- output is clamped, to never be bigger than +/- 8192. This is to prevent wind-up.
      
   pidPiezo: process (clk, reset, initOutput)
   variable p1                  : signed(35 downto 0);
   variable p                   : signed(60 downto 0);
   variable i                   : signed(60 downto 0);
   variable d1                  : signed(37 downto 0);
   variable d                   : signed(62 downto 0);
   variable unClamped           : signed(69 downto 0);
   variable currentError        : signed(35 downto 0);
   variable lastInput           : signed(35 downto 0);
   variable beforeLastInput     : signed(35 downto 0);
   variable currentSetpoint     : signed(35 downto 0);
   variable currentInput        : signed(35 downto 0);
   variable lastOutput          : signed(35 downto 0);
   variable pIDSum              : signed(44 downto 0);
   variable pIDSumWithGain      : signed(69 downto 0);

   variable lastError           : signed(35 downto 0);
   variable beforeLastError     : signed(35 downto 0);
   
   variable cntr                : integer := 0;
   
   begin
      if reset = '1' then
         currentInput := signed(input(input'left) & input & b"000_0000_0000_0000_0000");
         lastInput := currentInput;
         currentOutput <= signed(initOutput(initOutput'left) & initOutput & b"000_0000_0000_0000_0000");
         currentError := (OTHERS=>'0');
         cntr := 0;
      
      else
         if rising_edge(clk) then
            case cntr is
               when 0 =>
                  -- I follow Type C PID equation, where the setpoint only enters through
                  -- the I loop.      
                  beforeLastInput := lastInput;
                  lastInput := currentInput;
                  lastError := currentError;

                  lastOutput := currentOutput;

                  currentInput := signed(input(input'left) & input & b"000_0000_0000_0000_0000");
                  currentSetpoint := signed(setPoint(setPoint'left) & setPoint & b"000_0000_0000_0000_0000");
                  
                  currentError := currentSetpoint - currentInput;
            
                  cntr := 1;
         
               when 1 =>
                  -- TYPE A & B
                  -- p1 := (currentError - lastError);
                  -- TYPE C
                  p1 := (currentInput - lastInput);
                  cntr := cntr + 1;
               
               when 2 =>
                  p  := Kp * p1;
                  cntr := cntr + 1;
                  
               when 3 =>
                  p  := shr(p, conv_unsigned(19,61));
                  cntr := cntr + 1;
                  
               when 4 =>
                  i := Ki * (currentError);
                  cntr := cntr + 1;
                  
               when 5 =>
                  i := shr(i, conv_unsigned(19,61));
                  cntr := cntr + 1;
                  
               when 6 =>
                  -- TYPE A
                  -- d1 := (currentError - (conv_signed(2,2)*lastError) + beforeLastError);
                  -- TYPE B & C
                  d1 := (currentInput - (conv_signed(2,2)*lastInput) + beforeLastInput);
                  cntr := cntr + 1;
               
               when 7 =>
                  d := Kd * d1;
                  cntr := cntr + 1;
               
               when 8 =>
                  d := shr(d, conv_unsigned(19,63));
                  cntr := cntr + 1;
                  
               when 9 =>
                  -- TYPE A
                  -- pIDSum := conv_signed(0,43)-p(42 downto 0)-i(42 downto 0)-d(44 downto 0);
                  -- TYPE B
                  -- pIDSum := conv_signed(0,43)-p(42 downto 0)-i(42 downto 0)+d(44 downto 0);
                  -- TYPE C
                  pIDSum := p(42 downto 0)-i(42 downto 0)+d(44 downto 0);
                  cntr := cntr + 1;
                  
               when 10 =>
                  pIDSumWithGain := gain * pIDSum;
                  cntr := cntr + 1;
              
               when 11 =>
                  unclamped := lastOutput-shr(pIDSumWithGain,conv_unsigned(12,70));
                  cntr := cntr + 1;
                  
               when 12 =>
                  -- clamp the output at min=-8192*2^11 and max=8191*2^11 to prevent
                  -- PID wind-up. 
                  if unclamped(69 downto 32)<conv_signed(-1,38) then -- if smaller than -8192, then clamp at -8192
                     currentOutput(35 downto 19) <= conv_signed(-8192,17);
                     currentOutput(18 downto 0) <= (OTHERS=>'0');
                  elsif unclamped(69 downto 32)>conv_signed(0,38) then  -- if > 8191, clamp at 8191
                     currentOutput(35 downto 19) <= conv_signed(8191,17);
                     currentOutput(18 downto 0) <= (OTHERS=>'0');
                  else
                     currentOutput(35 downto 0) <= unclamped(35 downto 0); -- else keep value
                  end if;
                  cntr := 0;
                  
               when others =>
                  cntr := 0;
            end case;
         end if;
      end if;
   end process;
   
 -- convert to 16bit and output
 
   output <= (OTHERS=>'0') when bypass = '1' else 
      currentOutput(34 downto 19);

   
end Behavioral;