----------------------------------------------------------------------------
-- DPIMREF.VHD -- Digilent Parallel Interface Module Reference Design
----------------------------------------------------------------------------
-- Author:  Gene Apperson
--          Copyright 2004 Digilent, Inc.
----------------------------------------------------------------------------
-- IMPORTANT NOTE ABOUT BUILDING THIS LOGIC IN ISE
--
-- Before building the Dpimref logic in ISE:
--    1.    In Project Navigator, right-click on "Synthesize-XST"
--    (in the Process View Tab) and select "Properties"
-- 2. Click the "HDL Options" tab
-- 3.    Set the "FSM Encoding Algorithm" to "None"
----------------------------------------------------------------------------
--
----------------------------------------------------------------------------
-- This module contains an example implementation of Digilent Parallel
-- Interface Module logic. This interface is used in conjunction with the
-- DPCUTIL DLL and a Digilent Communications Module (USB, EtherNet, Serial)
-- to exchange data with an application running on a host PC and the logic
-- implemented in a gate array.
--
-- See the Digilent document, Digilent Parallel Interface Model Reference
-- Manual (doc # 560-000) for a description of the interface.
--
-- This design uses a state machine implementation to respond to transfer
-- cycles. It implements an address register, 8 internal data registers
-- that merely hold a value written, and interface registers to communicate
-- with a Digilent DIO4 board. There is an LED output register whose value 
-- drives the 8 discrete leds on the DIO4. There are two input registers.
-- One reads the switches on the DIO4 and the other reads the buttons.
--
-- Interface signals used in top level entity port:
--    mclk     - master clock, generally 50Mhz osc on system board
--    pdb         - port data bus
--    astb     - address strobe
--    dstb     - data strobe
--    pwr         - data direction (described in reference manual as WRITE)
--    pwait    - transfer synchronization (described in reference manual
--                as WAIT)
--    rgLed    - LED outputs to the DIO4
--    rgSwt    - switch inputs from the DIO4
--    ldb         - led gate signal for the DIO4
--    rgBtn    - button inputs from the DIO4
--    btn         - button on system board (D2SB or D2FT)
--    led         - led on the system board
----------------------------------------------------------------------------
-- Revision History:
--  06/09/2004(GeneA): created
-- 08/10/2004(GeneA): initial public release
-- 04/25/2006(JoshP): comment addition
-- 02/14/2010(Arne): Changed source to have 32 (#0-#31) single byte data 
--    registers instead of just 8. Register #29 is read-only and reads the 
--    status of the 8 onboard switches, register #30 is read-only and reads
--    the status of the 4 push buttons, register #31 is write-only and
--    writes to the onboard LEDS
-- 03/18/2011 (Arne): Changed source to have 50 (#0-#49) single byte data
--    registers. Register #50 is read-only and reads the status of the
--    8 onboard switches "rgSwt," register #51 is read only and reads the
--    status of the 4 push buttons "rgBtn"
----------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

--  Uncomment the following lines to use the declarations that are
--  provided for instantiating Xilinx primitive components.
--library UNISIM;
--use UNISIM.VComponents.all;

entity dpimref_modified is
    Port (
      mclk  :  in std_logic;
      pdb   :  inout std_logic_vector(7 downto 0);
      astb  :  in std_logic;
      dstb  :  in std_logic;
      pwr   :  in std_logic;
      pwait :  out std_logic;
      rgLed :  out std_logic_vector(7 downto 0); 
      rgSwt :  in std_logic_vector(7 downto 0);
      rgBtn :  in std_logic_vector(3 downto 0);
      -- btn      : in std_logic;
      -- ldg   :  out std_logic;
      -- led      : out std_logic;
      register00  : out std_logic_vector(7 downto 0);
      register01  : out std_logic_vector(7 downto 0);
      register02  : out std_logic_vector(7 downto 0);
      register03  : out std_logic_vector(7 downto 0);
      register04  : out std_logic_vector(7 downto 0);
      register05  : out std_logic_vector(7 downto 0);
      register06  : out std_logic_vector(7 downto 0);
      register07  : out std_logic_vector(7 downto 0);
      register08  : out std_logic_vector(7 downto 0);
      register09  : out std_logic_vector(7 downto 0);
      register10  : out std_logic_vector(7 downto 0);
      register11  : out std_logic_vector(7 downto 0);
      register12  : out std_logic_vector(7 downto 0);
      register13  : out std_logic_vector(7 downto 0);
      register14  : out std_logic_vector(7 downto 0);
      register15  : out std_logic_vector(7 downto 0);
      register16  : out std_logic_vector(7 downto 0);
      register17  : out std_logic_vector(7 downto 0);
      register18  : out std_logic_vector(7 downto 0);
      register19  : out std_logic_vector(7 downto 0);
      register20  : out std_logic_vector(7 downto 0);
      register21  : out std_logic_vector(7 downto 0);
      register22  : out std_logic_vector(7 downto 0);
      register23  : out std_logic_vector(7 downto 0);
      register24  : out std_logic_vector(7 downto 0);
      register25  : out std_logic_vector(7 downto 0);
      register26  : out std_logic_vector(7 downto 0);
      register27  : out std_logic_vector(7 downto 0);
      register28  : out std_logic_vector(7 downto 0);
      register29  : out std_logic_vector(7 downto 0);
      register30  : out std_logic_vector(7 downto 0);
      register31  : out std_logic_vector(7 downto 0);
      register32  : out std_logic_vector(7 downto 0);
      register33  : out std_logic_vector(7 downto 0);
      register34  : out std_logic_vector(7 downto 0);
      register35  : out std_logic_vector(7 downto 0);
      register36  : out std_logic_vector(7 downto 0);
      register37  : out std_logic_vector(7 downto 0);
      register38  : out std_logic_vector(7 downto 0);
      register39  : out std_logic_vector(7 downto 0);
      register40  : out std_logic_vector(7 downto 0);
      register41  : out std_logic_vector(7 downto 0);
      register42  : out std_logic_vector(7 downto 0);
      register43  : out std_logic_vector(7 downto 0);
      register44  : out std_logic_vector(7 downto 0);
      register45  : out std_logic_vector(7 downto 0);
      register46  : out std_logic_vector(7 downto 0);
      register47  : out std_logic_vector(7 downto 0);
      register48  : out std_logic_vector(7 downto 0);
      register49  : out std_logic_vector(7 downto 0)
   );
end dpimref_modified;

architecture Behavioral of dpimref_modified is

------------------------------------------------------------------------
-- Component Declarations
------------------------------------------------------------------------

------------------------------------------------------------------------
-- Local Type Declarations
------------------------------------------------------------------------

------------------------------------------------------------------------
--  Constant Declarations
------------------------------------------------------------------------

   -- The following constants define state codes for the EPP port interface
   -- state machine. The high order bits of the state number give a unique
   -- state identifier. The low order bits are the state machine outputs for
   -- that state. This type of state machine implementation uses no
   -- combination logic to generate outputs which should produce glitch
   -- free outputs.
   constant stEppReady  : std_logic_vector(7 downto 0) := "0000" & "0000";
   constant stEppAwrA   : std_logic_vector(7 downto 0) := "0001" & "0100";
   constant stEppAwrB   : std_logic_vector(7 downto 0) := "0010" & "0001";
   constant stEppArdA   : std_logic_vector(7 downto 0) := "0011" & "0010";
   constant stEppArdB   : std_logic_vector(7 downto 0) := "0100" & "0011";
   constant stEppDwrA   : std_logic_vector(7 downto 0) := "0101" & "1000";
   constant stEppDwrB   : std_logic_vector(7 downto 0) := "0110" & "0001";
   constant stEppDrdA   : std_logic_vector(7 downto 0) := "0111" & "0010";
   constant stEppDrdB   : std_logic_vector(7 downto 0) := "1000" & "0011";
   
   
------------------------------------------------------------------------
-- Signal Declarations
------------------------------------------------------------------------

   -- State machine current state register
   signal   stEppCur    : std_logic_vector(7 downto 0) := stEppReady;

   signal   stEppNext   : std_logic_vector(7 downto 0);

   signal   clkMain     : std_logic;

   -- Internal control signales
   signal   ctlEppWait  : std_logic;
   signal   ctlEppAstb  : std_logic;
   signal   ctlEppDstb  : std_logic;
   signal   ctlEppDir   : std_logic;
   signal   ctlEppWr    : std_logic;
   signal   ctlEppAwr   : std_logic;
   signal   ctlEppDwr   : std_logic;
   signal   busEppOut   : std_logic_vector(7 downto 0);
   signal   busEppIn    : std_logic_vector(7 downto 0);
   signal   busEppData  : std_logic_vector(7 downto 0);

   -- Registers
   signal   regEppAdr   : std_logic_vector(5 downto 0);
   signal   regData0    : std_logic_vector(7 downto 0);
   signal   regData1    : std_logic_vector(7 downto 0);
   signal   regData2    : std_logic_vector(7 downto 0);
   signal   regData3    : std_logic_vector(7 downto 0);
   signal   regData4    : std_logic_vector(7 downto 0);
   signal   regData5    : std_logic_vector(7 downto 0);
   signal   regData6    : std_logic_vector(7 downto 0);
   signal   regData7    : std_logic_vector(7 downto 0);
   signal   regData8    : std_logic_vector(7 downto 0);
   signal   regData9    : std_logic_vector(7 downto 0);
   signal   regData10   : std_logic_vector(7 downto 0);
   signal   regData11   : std_logic_vector(7 downto 0);
   signal   regData12   : std_logic_vector(7 downto 0);
   signal   regData13   : std_logic_vector(7 downto 0);
   signal   regData14   : std_logic_vector(7 downto 0);
   signal   regData15   : std_logic_vector(7 downto 0);
   signal   regData16   : std_logic_vector(7 downto 0);
   signal   regData17   : std_logic_vector(7 downto 0);
   signal   regData18   : std_logic_vector(7 downto 0);
   signal   regData19   : std_logic_vector(7 downto 0);
   signal   regData20   : std_logic_vector(7 downto 0);
   signal   regData21   : std_logic_vector(7 downto 0);
   signal   regData22   : std_logic_vector(7 downto 0);
   signal   regData23   : std_logic_vector(7 downto 0);
   signal   regData24   : std_logic_vector(7 downto 0);
   signal   regData25   : std_logic_vector(7 downto 0);
   signal   regData26   : std_logic_vector(7 downto 0);
   signal   regData27   : std_logic_vector(7 downto 0);
   signal   regData28   : std_logic_vector(7 downto 0);
   signal   regData29   : std_logic_vector(7 downto 0);
   signal   regData30   : std_logic_vector(7 downto 0);
   signal   regData31   : std_logic_vector(7 downto 0);
   signal   regData32   : std_logic_vector(7 downto 0);
   signal   regData33   : std_logic_vector(7 downto 0);
   signal   regData34   : std_logic_vector(7 downto 0);
   signal   regData35   : std_logic_vector(7 downto 0);
   signal   regData36   : std_logic_vector(7 downto 0);
   signal   regData37   : std_logic_vector(7 downto 0);
   signal   regData38   : std_logic_vector(7 downto 0);
   signal   regData39   : std_logic_vector(7 downto 0);
   signal   regData40   : std_logic_vector(7 downto 0);
   signal   regData41   : std_logic_vector(7 downto 0);
   signal   regData42   : std_logic_vector(7 downto 0);
   signal   regData43   : std_logic_vector(7 downto 0);
   signal   regData44   : std_logic_vector(7 downto 0);
   signal   regData45   : std_logic_vector(7 downto 0);
   signal   regData46   : std_logic_vector(7 downto 0);
   signal   regData47   : std_logic_vector(7 downto 0);
   signal   regData48   : std_logic_vector(7 downto 0);
   signal   regData49   : std_logic_vector(7 downto 0);

   signal   regLed      : std_logic_vector(7 downto 0);

   signal   cntr        : std_logic_vector(23 downto 0);


------------------------------------------------------------------------
-- Module Implementation
------------------------------------------------------------------------

begin

    ------------------------------------------------------------------------
   -- Map basic status and control signals
    ------------------------------------------------------------------------

   clkMain <= mclk;

   ctlEppAstb <= astb;
   ctlEppDstb <= dstb;
   ctlEppWr   <= pwr;
   pwait      <= ctlEppWait;  -- drive WAIT from state machine output

   -- Data bus direction control. The internal input data bus always
   -- gets the port data bus. The port data bus drives the internal
   -- output data bus onto the pins when the interface says we are doing
   -- a read cycle and we are in one of the read cycles states in the
   -- state machine.
   busEppIn <= pdb;
   pdb <= busEppOut when ctlEppWr = '1' and ctlEppDir = '1' else "ZZZZZZZZ";

   -- Select either address or data onto the internal output data bus.
   -- busEppOut <= "0000" & regEppAdr when ctlEppAstb = '0' else busEppData;
   busEppOut <= "00" & regEppAdr when ctlEppAstb = '0' else busEppData;


   rgLed <= regLed;
   -- ldg <= '1';

   -- Decode the address register and select the appropriate data register
   busEppData <=  regData0 when regEppAdr = "000000" else
               regData1 when regEppAdr = "000001" else
               regData2 when regEppAdr = "000010" else
               regData3 when regEppAdr = "000011" else
               regData4 when regEppAdr = "000100" else
               regData5 when regEppAdr = "000101" else
               regData6 when regEppAdr = "000110" else
               regData7 when regEppAdr = "000111" else
               regData8 when regEppAdr = "001000" else
               regData9 when regEppAdr = "001001" else
               regData10 when regEppAdr = "001010" else
               regData11 when regEppAdr = "001011" else
               regData12 when regEppAdr = "001100" else
               regData13 when regEppAdr = "001101" else
               regData14 when regEppAdr = "001110" else
               regData15 when regEppAdr = "001111" else
               regData16 when regEppAdr = "010000" else
               regData17 when regEppAdr = "010001" else
               regData18 when regEppAdr = "010010" else
               regData19 when regEppAdr = "010011" else
               regData20 when regEppAdr = "010100" else
               regData21 when regEppAdr = "010101" else
               regData22 when regEppAdr = "010110" else
               regData23 when regEppAdr = "010111" else
               regData24 when regEppAdr = "011000" else
               regData25 when regEppAdr = "011001" else
               regData26 when regEppAdr = "011010" else
               regData27 when regEppAdr = "011011" else
               regData28 when regEppAdr = "011100" else
               regData29 when regEppAdr = "011101" else
               regData30 when regEppAdr = "011110" else
               regData31 when regEppAdr = "011111" else
               regData32 when regEppAdr = "100000" else
               regData33 when regEppAdr = "100001" else
               regData34 when regEppAdr = "100010" else
               regData35 when regEppAdr = "100011" else
               regData36 when regEppAdr = "100100" else
               regData37 when regEppAdr = "100101" else
               regData38 when regEppAdr = "100110" else
               regData39 when regEppAdr = "100111" else
               regData40 when regEppAdr = "101000" else
               regData41 when regEppAdr = "101001" else
               regData42 when regEppAdr = "101010" else
               regData43 when regEppAdr = "101011" else
               regData44 when regEppAdr = "101100" else
               regData45 when regEppAdr = "101101" else
               regData46 when regEppAdr = "101110" else
               regData47 when regEppAdr = "101111" else
               regData48 when regEppAdr = "110000" else
               regData49 when regEppAdr = "110001" else
               rgSwt    when regEppAdr = "110010" else
               "0000" & rgBtn when regEppAdr = "110011" else
               "00000000";

    ------------------------------------------------------------------------
   -- EPP Interface Control State Machine
    ------------------------------------------------------------------------

   -- Map control signals from the current state
   ctlEppWait <= stEppCur(0);
   ctlEppDir  <= stEppCur(1);
   ctlEppAwr  <= stEppCur(2);
   ctlEppDwr  <= stEppCur(3);

   -- This process moves the state machine to the next state
   -- on each clock cycle
   process (clkMain)
      begin
         if clkMain = '1' and clkMain'Event then
            stEppCur <= stEppNext;
         end if;
      end process;

   -- This process determines the next state machine state based
   -- on the current state and the state machine inputs.
   process (stEppCur, stEppNext, ctlEppAstb, ctlEppDstb, ctlEppWr)
      begin
         case stEppCur is
            -- Idle state waiting for the beginning of an EPP cycle
            when stEppReady =>
               if ctlEppAstb = '0' then
                  -- Address read or write cycle
                  if ctlEppWr = '0' then
                     stEppNext <= stEppAwrA;
                  else
                     stEppNext <= stEppArdA;
                  end if;

               elsif ctlEppDstb = '0' then
                  -- Data read or write cycle
                  if ctlEppWr = '0' then
                     stEppNext <= stEppDwrA;
                  else
                     stEppNext <= stEppDrdA;
                  end if;

               else
                  -- Remain in ready state
                  stEppNext <= stEppReady;
               end if;                                

            -- Write address register
            when stEppAwrA =>
               stEppNext <= stEppAwrB;

            when stEppAwrB =>
               if ctlEppAstb = '0' then
                  stEppNext <= stEppAwrB;
               else
                  stEppNext <= stEppReady;
               end if;     

            -- Read address register
            when stEppArdA =>
               stEppNext <= stEppArdB;

            when stEppArdB =>
               if ctlEppAstb = '0' then
                  stEppNext <= stEppArdB;
               else
                  stEppNext <= stEppReady;
               end if;

            -- Write data register
            when stEppDwrA =>
               stEppNext <= stEppDwrB;

            when stEppDwrB =>
               if ctlEppDstb = '0' then
                  stEppNext <= stEppDwrB;
               else
                  stEppNext <= stEppReady;
               end if;

            -- Read data register
            when stEppDrdA =>
               stEppNext <= stEppDrdB;
                              
            when stEppDrdB =>
               if ctlEppDstb = '0' then
                  stEppNext <= stEppDrdB;
               else
                  stEppNext <= stEppReady;
               end if;

            -- Some unknown state            
            when others =>
               stEppNext <= stEppReady;

         end case;
      end process;
      
    ------------------------------------------------------------------------
   -- EPP Address register
    ------------------------------------------------------------------------

   process (clkMain, ctlEppAwr)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppAwr = '1' then
               regEppAdr <= busEppIn(5 downto 0);
            end if;
         end if;
      end process;

    ------------------------------------------------------------------------
   -- EPP Data registers
    ------------------------------------------------------------------------
   -- The following processes implement the interface registers. These
   -- registers just hold the value written so that it can be read back.
   -- In a real design, the contents of these registers would drive additional
   -- logic.
   -- The ctlEppDwr signal is an output from the state machine that says
   -- we are in a 'write data register' state. This is combined with the
   -- address in the address register to determine which register to write.

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000000" then
               regData0 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000001" then
               regData1 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000010" then
               regData2 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000011" then
               regData3 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000100" then
               regData4 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000101" then
               regData5 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000110" then
               regData6 <= busEppIn;
            end if;
         end if;
      end process;

   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "000111" then
               regData7 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001000" then
               regData8 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001001" then
               regData9 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001010" then
               regData10 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001011" then
               regData11 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001100" then
               regData12 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001101" then
               regData13 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001110" then
               regData14 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "001111" then
               regData15 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010000" then
               regData16 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010001" then
               regData17 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010010" then
               regData18 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010011" then
               regData19 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010100" then
               regData20 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010101" then
               regData21 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010110" then
               regData22 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "010111" then
               regData23 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011000" then
               regData24 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011001" then
               regData25 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011010" then
               regData26 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011011" then
               regData27 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011100" then
               regData28 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011101" then
               regData29 <= busEppIn;
            end if;
         end if;
      end process;


   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011110" then
               regData30 <= busEppIn;
            end if;
         end if;
      end process;


   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "011111" then
               regData31 <= busEppIn;
            end if;
         end if;
      end process;
   
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100000" then
               regData32 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100001" then
               regData33 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100010" then
               regData34 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100011" then
               regData35 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100100" then
               regData36 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100101" then
               regData37 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100110" then
               regData38 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "100111" then
               regData39 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101000" then
               regData40 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101001" then
               regData41 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101010" then
               regData42 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101011" then
               regData43 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101100" then
               regData44 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101101" then
               regData45 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101110" then
               regData46 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "101111" then
               regData47 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "110000" then
               regData48 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "110001" then
               regData49 <= busEppIn;
            end if;
         end if;
      end process;
      
   process (clkMain, regEppAdr, ctlEppDwr, busEppIn)
      begin
         if clkMain = '1' and clkMain'Event then
            if ctlEppDwr = '1' and regEppAdr = "110100" then
               regLED <= busEppIn;
            end if;
         end if;
      end process;

   register00 <= regData0;
   register01 <= regData1;
   register02 <= regData2;
   register03 <= regData3;
   register04 <= regData4;
   register05 <= regData5;
   register06 <= regData6;
   register07 <= regData7;
   register08 <= regData8;
   register09 <= regData9;
   register10 <= regData10;
   register11 <= regData11;
   register12 <= regData12;
   register13 <= regData13;
   register14 <= regData14;
   register15 <= regData15;
   register16 <= regData16;
   register17 <= regData17;
   register18 <= regData18;
   register19 <= regData19;
   register20 <= regData20;
   register21 <= regData21;
   register22 <= regData22;
   register23 <= regData23;
   register24 <= regData24;
   register25 <= regData25;
   register26 <= regData26;
   register27 <= regData27;
   register28 <= regData28;
   register29 <= regData29;
   register30 <= regData30;
   register31 <= regData31;
   register32 <= regData32;
   register33 <= regData33;
   register34 <= regData34;
   register35 <= regData35;
   register36 <= regData36;
   register37 <= regData37;
   register38 <= regData38;
   register39 <= regData39;
   register40 <= regData40;
   register41 <= regData41;
   register42 <= regData42;
   register43 <= regData43;
   register44 <= regData44;
   register45 <= regData45;
   register46 <= regData46;
   register47 <= regData47;
   register48 <= regData48;
   register49 <= regData49;

end Behavioral;
