----------------------------------------------------------------------------------
-- Company: The University of Oklahoma
-- Engineer: Arne Schwettmann
-- 
-- Create Date:    18:01:58 03/01/2010 
-- Design Name: 
-- Module Name:    highPass16bit - Behavioral 
-- Project Name:   Shaffer Lab Lockbox
-- Target Devices: Digilent Nexys2 1200k Spartan3 based evaluation board
-- Tool versions:
-- Description: 
--    please see the top level source file "lockbox.vhdl" for release info 
--
-- Dependencies: 
--
-- Revision: 
--    1.0 public release (August 2011)
--
-- Additional Comments: 
--
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity highPass16bit is
   port (
      clk      : in std_logic;
      cutOffFreqHz   : in unsigned(23 downto 0);
      input    : in signed(15 downto 0);
      output   : out signed(15 downto 0);
      bypass   : in std_logic;
      reset    : in std_logic
   );
end highPass16bit;

architecture Behavioral of highPass16bit is

   signal currentOutput : signed(35 downto 0);
 
begin

   -- implementation of discrete time high pass filter as described in 
   -- http://en.wikipedia.org/wiki/High-pass_filter
   -- y_n=alpha*y_(n-1)+alpha*(x_n-x_(n-1))
   -- here alpha is the smoothing time constant alpha=RC/(RC+DT)
   -- where RC is related to the cut off frequency in Hertz by
   -- RC=1/(2 pi f). For f much smaller than clk freq, alpha=1-2 pi f DT
   -- define beta=2 pi f DT then alpha=1-beta, and
   -- y_n=(1-beta)*y_(n-1)+(1-beta)*(x_n-x_(n-1))
   -- since the effective clkfreq is 50MHz/13, beta is 
   -- 6.28*1/clkfreq*f, a multiplication
   -- with beta is a shift of 19 bits to 
   -- the right then multiply by the cut off frequency

   -- sequential process calculates filtered signal
   discreteHighPass: process (clk, reset)
   variable intermediateResult1   : signed(35 downto 0);
   variable intermediateResult2   : signed(60 downto 0);
   variable lastInputMultiplied   : signed(60 downto 0);
   variable lastOutputMultiplied  : signed(60 downto 0);
   variable currentInputMultiplied   : signed(60 downto 0);
   variable lastOutput           : signed(35 downto 0);
   variable currentInput         : signed(35 downto 0);
   variable lastInput            : signed(35 downto 0);
   variable cntr                 : integer := 0;
   begin
      if reset='1' then
         currentOutput <= conv_signed(0,36);
         currentInput := conv_signed(0,36);
         cntr := 0;
      else
         if rising_edge(clk) then
            case cntr is
            
               when 0 =>
                        lastOutput := currentOutput;
                        lastInput := currentInput;
                        if input(input'left)='0' then 
                           currentInput := input & b"0000_0000_0000_0000_0000";
                        else
                           currentInput := input & b"0000_0000_0000_0000_0000";
                        end if;
                        cntr := 1;
               
               when 1 =>
                  intermediateResult1 := shr(lastInput, conv_unsigned(19,36));
                  cntr := cntr + 1;
                  
               when 2 =>
                  lastInputMultiplied := intermediateResult1 * cutOffFreqHz;
                  cntr := cntr + 1;
                  
               when 3 =>
                  intermediateResult1 := shr(lastOutput, conv_unsigned(19,36));
                  cntr := cntr + 1;
                  
               when 4 => 
                  lastOutputMultiplied := intermediateResult1 * cutOffFreqHz;
                  cntr := cntr + 1;
                  
               when 5 =>
                  intermediateResult1 := shr(currentInput, conv_unsigned(19,36));
                  cntr := cntr + 1;
                  
               when 6 =>
                  currentInputMultiplied := intermediateResult1 * cutOffFreqHz;
                  cntr := cntr + 1;
               
               when 7 =>
                  intermediateResult2 := lastOutput + lastInputMultiplied;
                  cntr := cntr + 1;
                  
               when 8 =>
                  intermediateResult2 := intermediateResult2 + currentInput;
                  cntr := cntr + 1;
                  
               when 9 =>
                  intermediateResult2 := intermediateResult2 - lastInput;
                  cntr := cntr + 1;
                  
               when 10 =>
                  intermediateResult2 := intermediateResult2 - lastOutputMultiplied;
                  cntr := cntr + 1;
                  
               when 11 =>
                  intermediateResult2 := intermediateResult2 - currentInputMultiplied;
                  cntr := cntr + 1;
                   
               when 12 =>
                  currentOutput <= intermediateResult2(35 downto 0);
--            if intermediateResult<(-8192)*2**14 then -- if smaller than -8192, then clamp at -8192
--               currentOutput <= conv_signed(-8192*2**14,32);
--            elsif intermediateResult>(8191)*2**14 then  -- if > 8191, clamp at 8191?
--                  currentOutput <= conv_signed(8191*2**14,32);
--            else
--                  currentOutput <= intermediateResult(31 downto 0); -- else keep value
--            end if;
                  cntr := 0;
                  
               when others =>
                  cntr := 0;
            
            end case;
         end if;
      end if;
   end process;
   
-- convert to 16bit and output
    output <= input when bypass = '1' else 
      currentOutput(35 downto 20);
               
end Behavioral;
