----------------------------------------------------------------------------------
-- Company: The University of Oklahoma
-- Engineer: Arne Schwettmann
-- 
-- Create Date: 
--    13:10:21 02/21/2010 
-- Design Name: 
--    Shaffer Lockbox
-- Module Name: 
--    lockbox_toplevel - Behavioral 
-- Project Name: 
--    Shaffer Lab Lockbox
-- Target Devices: 
--    Digilent Nexys2 1200k Spartan3 based evaluation board
-- Tool versions: 
--
-- Copyright:
--    This program is public domain. However, he who makes use of this
--    program should give credit to the authors and cite the reference:
--
--    Arne Schwettmann, Jonathon Sedlacek, and James P. Shaffer,
--    "FPGA-based locking circuit for external cavity diode laser frequency
--    stabilization," Rev. Sci. Inst. (2011)
--
--    We will try to give limited support, but can not make any guarantees
--    contact: shaffer@nhn.ou.edu
--
-- Description: 
--    This is the Shaffer lab laser locking circuit top level VHDL source file.
--    It implements 2 PID loops, one for the current and one for the piezo.
--    PID parameters are controlled from a connected PC via USB.
--    USB communication is done via DPCUTIL C++ library and DPIMREF
--    loaded onto the FPGA. Analog in and out is done through Digilent PMOD
--    DA2 and AD2 modules.
--
-- Portability:
--    Use properties/hdl options/multiplies style="LUT" to compile.
--    The compiler used is XILINX WEBPACK ISE 10.1.
--
-- Written for:
--    Dr. James P. Shaffer's research group,
--    Homer L. Dodge Dept. of Physics
--    University of Oklahoma
--    e-mail: shaffer@nhn.ou.edu
--    More contact info can be found at www.nhn.ou.edu
--
-- Revision: 
--    1.0 public release (August 2011)
--
-- Additional Comments: 
--
-- Project Files:
--    lockbox.vhdl - top level source code, main program
--    lowPass16bit.vhd - source code for low pass filter module
--    highpass16bit.vhd - source code for high pass filter module
--    pidPiezo.vhd - source code for Piezo PID module
--    pidCurrent.vhd - source code for Current PID module
--    triangleGenerator16bit.vhd - source code for triangle wave generator module
--    display.vhd - source code for on-board seven segment display utilization
--    dpimref_mod.vhd - Digilent reference implementation of a parallel interface
--       for data exchange over USB, modified
--    AD1RefComp_mod.vhd - Digilent reference implementation for PmodAD1 
--       analog-to-digital input, modified
--    DA2RefComp_mod.vhd - Digilent reference implementation for PmodDA2 
--       digital-to-analog output peripheral module, modified
--    Nexys2-1200kLockbox.ucf - universal constraints file, contains timing
--       constraint and port-to-pin routing information for the Nexys2 board
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity lockbox is
   port(
      mclk  : in std_logic; -- master clock on Nexys 2 (50 MHz)
      pdb   : inout std_logic_vector(7 downto 0); -- used by DPIMREF (for USB communication)
      astb  : in std_logic; -- used by DPIMREF (for USB communication)
      dstb  : in std_logic; -- used by DPIMREF (for USB communication)
      pwr   : in std_logic; -- used by DPIMREF (for USB communication)
      pwait : out std_logic; -- used by DPIMREF (for USB communication)
      rgLed : out std_logic_vector(7 downto 0); -- on board LEDS
      rgSwt : in std_logic_vector(7 downto 0); -- on board flip switches
      rgBtn : in std_logic_vector(3 downto 0); -- on board buttons
      seg   : out std_logic_vector(6 downto 0); -- used by seven segment display
      an    : out std_logic_vector(3 downto 0); -- used by seven segment display
      dp    : out std_logic; -- used by seven segment display
      
      -- PMOD AD1 pins
      bitCsAdc    : out std_logic; --JA(0)
      bitD1Adc    : in std_logic; --JA(1)
      bitD2Adc    : in std_logic; --JA(2)
      bitSclkAdc  : out std_logic; --JA(3)
      
      -- PMOD DA2 pins
      bitnSyncDac : out std_logic; -- JB(0)
      bitD1DAC    : out std_logic; -- JB(1)
      bitD2DAC    : out std_logic; --JB(2)
      bitClkDac   : out std_logic --JB(3)           
   ); 
end lockbox;

architecture lockbox of lockbox is

-- first order 16bit low pass filter with asynchronous reset
-- internal bit-width is 36 bit to increase precision

component lowPass16bit is
   port (
      clk            : in std_logic;
      cutOffFreqHz   : in unsigned(23 downto 0);
      input          : in signed(15 downto 0);
      output         : out signed(15 downto 0);
      bypass         : in std_logic;
      reset          : in std_logic
   );
end component;

-- first order 16bit high pass filter with asynchronous reset
-- internal bit-width is 36 bit to increase precision

component highPass16bit is
   port (
      clk            : in std_logic;
      cutOffFreqHz   : in unsigned(23 downto 0);
      input          : in signed(15 downto 0);
      output         : out signed(15 downto 0);
      bypass         : in std_logic;
      reset          : in std_logic
   );
end component;

-- type C 16 bit PID loop with asynchronous reset
-- internal bit-width is 36 bit to increase precision

component pidPiezo is
   port (
      clk            : in std_logic;
      input    : in signed(15 downto 0);
      output   : out signed(15 downto 0);
      setPoint : in signed(15 downto 0);
      initOutput : in signed(15 downto 0);
      KP        : in unsigned(23 downto 0);
      KI        : in unsigned(23 downto 0);
      KD        : in unsigned(23 downto 0);
      gain     : in unsigned(23 downto 0);
      bypass   : in std_logic;
      reset    : in std_logic   
   );
end component;

-- type C 16 bit PID loop with asynchronous reset,
-- input polarity select, and high-pass filter on input
-- internal bit-width is 36 bit to increase precision

component pidCurrent is
 port (
      clk            : in std_logic;
      input    : in signed(15 downto 0);
      output   : out signed(15 downto 0);
      setPoint : in signed(15 downto 0);
      initOutput : in signed(15 downto 0);
      KP        : in unsigned(23 downto 0);
      KI        : in unsigned(23 downto 0);
      KD        : in unsigned(23 downto 0);
      gain     : in unsigned(23 downto 0);
      iCutOffFreqHz : in unsigned(15 downto 0);
      iCutOffBypass : in std_logic;
      polarity : in std_logic;
      bypass   : in std_logic;
      reset    : in std_logic      
  );
end component;

-- crude 16 bit triangle generator
-- deltay and deltax are the components of the slope 
-- slope is calculated by the user interface

component triangleGenerator16bit is
   port (
      clk      : in std_logic;
      output   : out signed(15 downto 0);
      deltaY   : in unsigned(15 downto 0);
      deltaX   : in unsigned(15 downto 0);
      amplitude: in unsigned(15 downto 0);
      constantOffset   : in signed(15 downto 0);
      bypass   : in std_logic;
      reset    : in std_logic
   );
end component;

-- encoder for on-board 7 segment display 
-- displays "numbertodisplay" as a decimal number on the on-board display

component segmentDisplay is
   port (
      clkMain  : in std_logic;
      seg      : out std_logic_vector(6 downto 0);
      an       : out std_logic_vector(3 downto 0);
      dp       : out std_logic;
      numberToDisplay : in std_logic_vector(11 downto 0) -- the number to display
   );
end component;

-- Dpimref_modified is the Digilent reference implementation of a parallel 
-- interface that is used for USB communication between desktop 
-- computer and FPGA board. 
-- Modified so that 50 registers are exposed to the rest 
-- of the program. The number of registers can be increased to 
-- a maximum of 256 if neccessary.

component dpimref_modified is
   port (
      mclk  : in std_logic;
      pdb   : inout std_logic_vector(7 downto 0);
      astb  : in std_logic;
      dstb  : in std_logic;
      pwr   : in std_logic;
      pwait : out std_logic;
      rgLed : out std_logic_vector(7 downto 0); 
      rgSwt : in std_logic_vector(7 downto 0);
      rgBtn : in std_logic_vector(3 downto 0);
      register00  : out std_logic_vector(7 downto 0); 
      register01  : out std_logic_vector(7 downto 0); 
      register02  : out std_logic_vector(7 downto 0); 
      register03  : out std_logic_vector(7 downto 0); 
      register04  : out std_logic_vector(7 downto 0); 
      register05  : out std_logic_vector(7 downto 0); 
      register06  : out std_logic_vector(7 downto 0); 
      register07  : out std_logic_vector(7 downto 0); 
      register08  : out std_logic_vector(7 downto 0); 
      register09  : out std_logic_vector(7 downto 0); 
      register10  : out std_logic_vector(7 downto 0); 
      register11  : out std_logic_vector(7 downto 0); 
      register12  : out std_logic_vector(7 downto 0); 
      register13  : out std_logic_vector(7 downto 0); 
      register14  : out std_logic_vector(7 downto 0);         
      register15  : out std_logic_vector(7 downto 0);         
      register16  : out std_logic_vector(7 downto 0);         
      register17  : out std_logic_vector(7 downto 0);         
      register18  : out std_logic_vector(7 downto 0);         
      register19  : out std_logic_vector(7 downto 0);         
      register20  : out std_logic_vector(7 downto 0);         
      register21  : out std_logic_vector(7 downto 0);         
      register22  : out std_logic_vector(7 downto 0);         
      register23  : out std_logic_vector(7 downto 0);         
      register24  : out std_logic_vector(7 downto 0);         
      register25  : out std_logic_vector(7 downto 0);         
      register26  : out std_logic_vector(7 downto 0);         
      register27  : out std_logic_vector(7 downto 0);      
      register28  : out std_logic_vector(7 downto 0); 
      register29  : out std_logic_vector(7 downto 0);         
      register30  : out std_logic_vector(7 downto 0); 
      register31  : out std_logic_vector(7 downto 0);         
      register32  : out std_logic_vector(7 downto 0);         
      register33  : out std_logic_vector(7 downto 0);         
      register34  : out std_logic_vector(7 downto 0);         
      register35  : out std_logic_vector(7 downto 0);         
      register36  : out std_logic_vector(7 downto 0);         
      register37  : out std_logic_vector(7 downto 0);         
      register38  : out std_logic_vector(7 downto 0);         
      register39  : out std_logic_vector(7 downto 0);         
      register40  : out std_logic_vector(7 downto 0);         
      register41  : out std_logic_vector(7 downto 0);         
      register42  : out std_logic_vector(7 downto 0);         
      register43  : out std_logic_vector(7 downto 0);         
      register44  : out std_logic_vector(7 downto 0);         
      register45  : out std_logic_vector(7 downto 0);         
      register46  : out std_logic_vector(7 downto 0);         
      register47  : out std_logic_vector(7 downto 0);         
      register48  : out std_logic_vector(7 downto 0);         
      register49  : out std_logic_vector(7 downto 0)          
  );
end component;

-- AD1_controller_modified is the Digilent reference implementation for PMOD-AD1 12 bit output module
-- slightly modified for continous operation with start high

component AD1_controller_modified is
   port (   
      --General usage
      CLK      : in std_logic;   -- System Clock (50MHz)     
      RST      : in std_logic;
      --Pmod interface signals
      SDATA1   : in std_logic;
      SDATA2   : in std_logic;
      SCLK     : out std_logic;
      CS       : out std_logic;
      --User interface signals
      DATA1    : out std_logic_vector(11 downto 0);
      DATA2    : out std_logic_vector(11 downto 0);
      START    : in std_logic; 
      DONE     : out std_logic   
   );
end component;

-- DA2_controller_modded is the Digilent reference implementation for PMOD-DA2 12 input bit module
-- slightly modified for continous operation with start always high

component DA2_controller_modified is
   port ( 
      --General usage
      CLK      : in std_logic;   -- System Clock (50MHz)     
      RST      : in std_logic;
       --Pmod interface signals
      D1       : out std_logic;
      D2       : out std_logic;
      CLK_OUT  : out std_logic;
      nSYNC    : out std_logic;
      --User interface signals
      DATA1    : in std_logic_vector(11 downto 0);
      DATA2    : in std_logic_vector(11 downto 0);
      START    : in std_logic; 
      DONE     : out std_logic    
   );
end component;


signal numberToDisplay  : STD_LOGIC_VECTOR(11 downto 0); -- number that will be displayed on 7 segment display 

-- These are the registers that are written and read via USB by a C++ program running on 
-- the host computer (that uses DCPUTIL.dll to communicate with the FPGA).
-- These registers are used to easily tune PID gains and set other parameters such as filter
-- frequencies etc. For numbers with more than 8 bits, multiple registers are concatenated,
-- where the most significant bits are first. E.g. 
-- pIDPiezoP = pIDPiezoPRegister1 & pIDPiezoPRegister2 & pIDPiezoPRegister3
-- Register28 and register 29 are unused.
-- For the relation between actual DPIMREF register numbers and these signals, see the
-- instantiation of DPIMREF below. 

signal   pIDPiezoPRegister1             : std_logic_vector(7 downto 0); -- PID Piezo P high bits                    
signal   pIDPiezoPRegister2             : std_logic_vector(7 downto 0); -- PID Piezo P medium bits                
signal   pIDPiezoPRegister3             : std_logic_vector(7 downto 0); -- PID Piezo P low bits                    
signal   pIDPiezoIRegister1             : std_logic_vector(7 downto 0); -- PID Piezo I high bits                  
signal   pIDPiezoIRegister2             : std_logic_vector(7 downto 0); -- PID Piezo I medium bits                 
signal   pIDPiezoIRegister3             : std_logic_vector(7 downto 0); -- PID Piezo I low bits                    
signal   pIDPiezoDRegister1             : std_logic_vector(7 downto 0); -- PID Piezo D high bits                   
signal   pIDPiezoDRegister2             : std_logic_vector(7 downto 0); -- PID Piezo D medium bits                 
signal   pIDPiezoDRegister3             : std_logic_vector(7 downto 0); -- PID Piezo D low bits                    
signal   pIDPiezoGRegister1             : std_logic_vector(7 downto 0); -- PID Piezo G high bits                   
signal   pIDPiezoGRegister2             : std_logic_vector(7 downto 0); -- PID Piezo G medium bits                 
signal   pIDPiezoGRegister3             : std_logic_vector(7 downto 0); -- PID Piezo G low bits                    
signal   pIDPiezoSetPointRegister1      : std_logic_vector(7 downto 0); -- PID Piezo Setpoint high bits           
signal   pIDPiezoSetPointRegister2      : std_logic_vector(7 downto 0); -- PID Piezo Setpoint low bits            
signal   pIDCurrentPRegister1           : std_logic_vector(7 downto 0); -- PID Current P high bits                
signal   pIDCurrentPRegister2           : std_logic_vector(7 downto 0); -- PID Current P medium bits              
signal   pIDCurrentPRegister3           : std_logic_vector(7 downto 0); -- PID Current P low bits                 
signal   pIDCurrentIRegister1           : std_logic_vector(7 downto 0); -- PID Current I high bits                
signal   pIDCurrentIRegister2           : std_logic_vector(7 downto 0); -- PID Current I medium bits              
signal   pIDCurrentIRegister3           : std_logic_vector(7 downto 0); -- PID Current I low bits                 
signal   pIDCurrentDRegister1           : std_logic_vector(7 downto 0); -- PID Current D high bits                
signal   pIDCurrentDRegister2           : std_logic_vector(7 downto 0); -- PID Current D medium bits              
signal   pIDCurrentDRegister3           : std_logic_vector(7 downto 0); -- PID Current D low bits                 
signal   pIDCurrentGRegister1           : std_logic_vector(7 downto 0); -- PID Current G high bits                
signal   pIDCurrentGRegister2           : std_logic_vector(7 downto 0); -- PID Current G medium bits              
signal   pIDCurrentGRegister3           : std_logic_vector(7 downto 0); -- PID Current G low bits                 
signal   pIDCurrentICutOffFreqRegister1 : std_logic_vector(7 downto 0); -- PID Current I Cut Off Freq Hz high bits
signal   pIDCurrentICutOffFreqRegister2 : std_logic_vector(7 downto 0); -- PID Current I Cut Off Freq Hz low bits 
signal   register28                     : std_logic_vector(7 downto 0);                                           
signal   bypasses                       : std_logic_vector(7 downto 0); -- Bypasses                                
signal   bypasses2                      : std_logic_vector(7 downto 0); -- Bypasses2                               
signal   cutOffFreq1Register1           : std_logic_vector(7 downto 0); -- reg_cutOffFreq1Register1                
signal   cutOffFreq1Register2           : std_logic_vector(7 downto 0); -- reg_cutOffFreq1Register2                
signal   cutOffFreq1Register3           : std_logic_vector(7 downto 0); -- reg_cutOffFreq1Register3                
signal   cutOffFreq2Register1           : std_logic_vector(7 downto 0); -- reg_cutOffFreq2Register1                
signal   cutOffFreq2Register2           : std_logic_vector(7 downto 0); -- reg_cutOffFreq2Register2                
signal   cutOffFreq2Register3           : std_logic_vector(7 downto 0); -- reg_cutOffFreq2Register3                
signal   inputOffsetRegister1           : std_logic_vector(7 downto 0); -- Input Offset high bits                  
signal   inputOffsetRegister2           : std_logic_vector(7 downto 0); -- Input Offset low bits                   
signal   inputGainRegister1             : std_logic_vector(7 downto 0); -- Input Gain high bits                    
signal   inputGainRegister2             : std_logic_vector(7 downto 0); -- Input Gain low bits                     
signal   scanDeltaYRegister1            : std_logic_vector(7 downto 0); -- Scan Delta Y high bits                 
signal   scanDeltaYRegister2            : std_logic_vector(7 downto 0); -- Scan Delta Y low bits                  
signal   scanDeltaXRegister1            : std_logic_vector(7 downto 0); -- Scan Delta X high bits                 
signal   scanDeltaXRegister2            : std_logic_vector(7 downto 0); -- Scan Delta X low bits                  
signal   scanOffsetRegister1            : std_logic_vector(7 downto 0); -- Scan Offset high bits                  
signal   scanOffsetRegister2            : std_logic_vector(7 downto 0); -- Scan Offset low bits                   
signal   scanAmplitudeRegister1         : std_logic_vector(7 downto 0); -- Scan Amplitude high bits               
signal   scanAmplitudeRegister2         : std_logic_vector(7 downto 0); -- Scan Amplitude low bits                
signal   register49                     : std_logic_vector(7 downto 0);                                           

constant lowPass1BypassBit              : integer := 0; -- bypasses(0)
constant highPass1BypassBit             : integer := 1; -- bypasses(1)
constant pIDPiezoBypassBit              : integer := 2; -- bypasses(2)
constant pIDCurrentBypassBit            : integer := 3; -- bypasses(3)
constant testWaveFormEnableBit          : integer := 4; -- bypasses(4)
constant resetBit                       : integer := 5; -- bypasses(5)
constant scanBypassBit                  : integer := 6; -- bypasses(6)
constant lockedBit                      : integer := 7; -- bypasses(7)

constant PIDCurrentICutOffBypassBit     : integer := 0; -- bypasses2(0)
constant doneWritingBit                 : integer := 1; -- bypasses2(1)
constant bypassAllBit                   : integer := 2; -- bypasses2(2)
constant PIDCurrentPolarityBit          : integer := 3; -- bypasses2(3)
constant scanSetPointBit                : integer := 4; -- bypasses2(4)
constant scanRestartBit                 : integer := 5; -- bypasses2(5)
                                                                                                                               
-- AD input voltages
signal inVoltage1    : std_logic_vector(11 downto 0);
signal inVoltage2    : std_logic_vector(11 downto 0);

-- DA output voltages
signal outVoltage1   : std_logic_vector(11 downto 0);
signal outVoltage2   : std_logic_vector(11 downto 0);

-- a generated test waveform (only used for debugging)
signal generatedVoltage1   : signed(15 downto 0);

-- internal signal paths
signal inputSignal   : signed(15 downto 0); -- 16 bit internal path input signal
signal inputOffset   : signed(15 downto 0); -- 16 bit internal input offset signal
signal setPoint      : signed(15 downto 0); -- PID setpoint set on user interface
signal errorSignal   : signed(15 downto 0); -- PID errorsignal
signal outputSignal1 : signed(15 downto 0); -- 16 bit internal path output signal
signal outputSignal2 : signed(15 downto 0); -- 16 bit internal path output signal 2
signal scanSignal    : signed(15 downto 0); -- 16 bit internal path for scanning signal
signal lockSignalPiezo    : signed(15 downto 0); -- 16 bit internal path for Piezo locking signal 
signal lockSignalCurrent  : signed(15 downto 0); -- 16 bit internal path for Current locking signal 

-- low pass1 signals
signal lowPass1In       : signed(15 downto 0);
signal lowPass1Out      : signed(15 downto 0);
signal lowPass1Bypass   : std_logic;
signal lowPass1CutOffFreq : unsigned(23 downto 0);

-- high pass1 signals
signal highPass1In         : signed(15 downto 0);
signal highPass1Out        : signed(15 downto 0);
signal highPass1Bypass     : std_logic;
signal highPass1CutOffFreq : unsigned(23 downto 0);

-- pid piezo signals
signal pidPiezo1In         : signed(15 downto 0);
signal pidPiezo1Out        : signed(15 downto 0);
signal pidPiezo1Setpoint   : signed(15 downto 0);
signal pidPiezo1InitOutput : signed(15 downto 0);
signal pidPiezo1Kp         : unsigned(23 downto 0);
signal pidPiezo1Ki         : unsigned(23 downto 0);
signal pidPiezo1Kd         : unsigned(23 downto 0);
signal pidPiezo1Gain       : unsigned(23 downto 0);
signal pidPiezo1Bypass     : std_logic;
signal pidPiezo1Reset      : std_logic;

-- pid current signals
signal pidCurrent1In         : signed(15 downto 0);
signal pidCurrent1Out        : signed(15 downto 0);
signal pidCurrent1Setpoint   : signed(15 downto 0);
signal pidCurrent1ICutOffFreq : unsigned(15 downto 0);
signal pidCurrent1ICutOffBypass : std_logic;
signal pidCurrent1Kp         : unsigned(23 downto 0);
signal pidCurrent1Ki         : unsigned(23 downto 0);
signal pidCurrent1Kd         : unsigned(23 downto 0);
signal pidCurrent1Gain       : unsigned(23 downto 0);
signal pidCurrent1Polarity   : std_logic;
signal pidCurrent1Bypass     : std_logic;

-- waveform generator signals for scanning
signal scanGenerator1DeltaY   : unsigned(15 downto 0);
signal scanGenerator1DeltaX   : unsigned(15 downto 0);
signal scanGenerator1Offset   : signed(15 downto 0);
signal scanGenerator1Amplitude: unsigned(15 downto 0);
signal scanReset              : std_logic;

-- other signals
signal doneWriting            : std_logic;


-- clocks/counters

signal clkSample   : std_logic; -- sample clock, used for scan signal generation
constant sampleClockDivisor   : integer := 50; -- divisor for sample Clock 50MHz/divisor=sample Clock freq.
signal cnt1 : integer range 0 to sampleClockDivisor; -- counter to generate clkSample

signal reset : std_logic; -- async. reset signal

-- states for state machine: this distinguishes "locking" from "scanning" state
type state_type is (scan, lock, bypassall);
signal present_state : state_type;

begin

-- instantiate entities of the various components
-- and map the input/output ports of those "black boxes"

lowPass1: lowPass16bit
   port map (
      clk      => mclk,
      input    => lowPass1In,
      output   => lowPass1Out,
      cutOffFreqHz   => lowPass1CutOffFreq,
      bypass   => lowPass1Bypass,
      reset    => reset
   );
   
 
highPass1: highPass16bit
   port map (
      clk      => mclk,
      input    => highPass1In,
      output   => highPass1Out,
      cutOffFreqHz   => highPass1CutOffFreq,
      bypass   => highPass1Bypass,
      reset    => reset
   );

pidPiezo1: pidPiezo 
   port map (
      clk      => mclk,
      input    => pidPiezo1In,
      output   => pidPiezo1Out,
      setPoint => pidPiezo1SetPoint,
      initOutput => pidPiezo1InitOutput,
      Kp       => pidPiezo1Kp,
      Ki       => pidPiezo1Ki,
      Kd       => pidPiezo1Kd,
      gain     => pidPiezo1Gain,
      bypass   => pidPiezo1Bypass,
      reset    => pidPiezo1Reset
   );

pidCurrent1: pidCurrent 
   port map (
      clk      => mclk,
      input    => pidCurrent1In,
      output   => pidCurrent1Out,
      setPoint => pidCurrent1SetPoint,
      initOutput => (OTHERS=>'0'),
      Kp       => pidCurrent1Kp,
      Ki       => pidCurrent1Ki,
      Kd       => pidCurrent1Kd,
      gain     => pidCurrent1Gain,
      bypass   => pidCurrent1Bypass,
      iCutOffFreqHz   => pidCurrent1ICutOffFreq,
      iCutOffBypass => pidCurrent1ICutOffBypass,
      polarity => pidCurrent1polarity,
      reset    => reset
   );

scanGenerator1: triangleGenerator16bit 
   port map (
      clk      => clkSample,
      output   => generatedVoltage1,
      deltaY   => scanGenerator1DeltaY,
      deltaX   => scanGenerator1DeltaX,
      amplitude => scanGenerator1Amplitude,
      constantOffset   => scanGenerator1Offset,
      bypass   => bypasses(scanBypassBit),
      reset    => scanReset
   );

segDis1: segmentDisplay
    port map (
      clkMain  => clkSample,
      seg      => seg,
      an       => an,
      dp       => dp,
      numberToDisplay   => numberToDisplay
   );

DPIM1: dpimref_modified
   port map (
      mclk  => mclk,
      pdb   => pdb,   -- port data bus
      astb  => astb, -- address strobe
      dstb  => dstb, -- data strobe
      pwr   => pwr,   -- data direction (described in reference manual as WRITE)
      pwait => pwait,  -- pwait transfer synchronization (described in reference manual as WAIT)
      rgLed => rgLed,  -- LEDs on board
      rgSwt => rgSwt,  -- swiches on board
      rgBtn => rgBtn,  -- pushbuttons on board
      register00  => pIDPiezoPRegister1,                
      register01  => pIDPiezoPRegister2,                
      register02  => pIDPiezoPRegister3,                
      register03  => pIDPiezoIRegister1,                
      register04  => pIDPiezoIRegister2,                
      register05  => pIDPiezoIRegister3,                
      register06  => pIDPiezoDRegister1,                
      register07  => pIDPiezoDRegister2,                
      register08  => pIDPiezoDRegister3,                
      register09  => pIDPiezoGRegister1,                
      register10  => pIDPiezoGRegister2,                
      register11  => pIDPiezoGRegister3,                
      register12  => pIDPiezoSetPointRegister1,         
      register13  => pIDPiezoSetPointRegister2,         
      register14  => pIDCurrentPRegister1,              
      register15  => pIDCurrentPRegister2,              
      register16  => pIDCurrentPRegister3,              
      register17  => pIDCurrentIRegister1,              
      register18  => pIDCurrentIRegister2,              
      register19  => pIDCurrentIRegister3,              
      register20  => pIDCurrentDRegister1,              
      register21  => pIDCurrentDRegister2,              
      register22  => pIDCurrentDRegister3,              
      register23  => pIDCurrentGRegister1,              
      register24  => pIDCurrentGRegister2,              
      register25  => pIDCurrentGRegister3,              
      register26  => pIDCurrentICutOffFreqRegister1,   
      register27  => pIDCurrentICutOffFreqRegister2,   
      register28  => register28,   
      register29  => bypasses,                          
      register30  => bypasses2,                    
      register31  => cutOffFreq1Register1,              
      register32  => cutOffFreq1Register2,              
      register33  => cutOffFreq1Register3,              
      register34  => cutOffFreq2Register1,              
      register35  => cutOffFreq2Register2,              
      register36  => cutOffFreq2Register3,              
      register37  => inputOffsetRegister1,              
      register38  => inputOffsetRegister2,              
      register39  => inputGainRegister1,                
      register40  => inputGainRegister2,                
      register41  => scanDeltaYRegister1,               
      register42  => scanDeltaYRegister2,               
      register43  => scanDeltaXRegister1,               
      register44  => scanDeltaXRegister2,               
      register45  => scanOffsetRegister1,               
      register46  => scanOffsetRegister2,               
      register47  => scanAmplitudeRegister1,            
      register48  => scanAmplitudeRegister2,            
      register49  => register49                         
   );

-- To obtain the maximum possible input sample rate of 781 kHz, "start" is permanently set to '1' here. 
-- In conjunction, two lines in the if statement starting with "if start = '0'" at the end of 
-- PmodAD1.vhd are commented out to force the chip to start a new conversion immediately after the last one.

AD1a: AD1_controller_modified
   port map (
      CLK      => mclk,
      RST      => '0',
      SDATA1   => bitD1Adc, --JA(1), 
      SDATA2   => bitD2Adc, --JA(2), 
      SCLK     => bitSclkAdc, --JA(3), 
      CS       => bitCsAdc, --JA(0), 
      DATA1    => inVoltage1,
      DATA2    => inVoltage2,
      START    => '1', -- set to clkSample & uncomment lines in PmodAD1.vhd to limit the input sample rate to 500 kHz
      DONE     => open
   );

-- To obtain the maximum possible output sample rate of 1.56 MHz, "start" is permanently set to '1' here. 
-- In conjunction, two lines in the if statement starting with "if start = '0'" at the end of 
-- PmodDA21.vhd are commented out to force the chip to start a new conversion immediately after the last one.

DA2a: DA2_controller_modified
   port map (
      CLK      => mclk,
      RST      => '0',
      D1       => bitD1DAC,
      D2       => bitD2DAC,
      CLK_OUT  => bitClkDac, --JB(3)
      nSYNC    => bitnSyncDac, --JB(0)
      DATA1    => outVoltage1, --JB(1)
      DATA2    => outVoltage2, --JB(2)
      START    => '1', -- set to clkSample & uncomment lines in PmodDA2.vhd to limit the output sample rate to 1 MHz
      DONE     => open
   );

   -- Debugging info on seven segment display:
   -- Display the input Voltage Nr. 1 on the seven segment display, for debugging purposes.
   -- It is a number between 0 and 4095, because the AD converter has 12 bit resolution. 
   -- If the user flips the rightmost on board switch, the maximum possible number is displayed 
   -- to verify the display works. Input Voltage must be between 0 and 3.3V!
   -- Depending on the flip switch settings, the PID parameters can also be
   -- displayed (lowest 12 bit only) for debugging purposes.

   numberToDisplay <=   conv_std_logic_vector(pIDPiezo1Gain, 12)  when rgSwt(3 downto 0) = "0001" else
                        conv_std_logic_vector(pIDPiezo1Kp, 12)    when rgSwt(3 downto 0) = "0010" else
                        conv_std_logic_vector(pIDPiezo1Ki, 12)    when rgSwt(3 downto 0) = "0011" else
                        conv_std_logic_vector(pIDPiezo1Kd, 12)    when rgSwt(3 downto 0) = "0100" else
                        conv_std_logic_vector(pIDCurrent1Gain, 12)  when rgSwt(3 downto 0) = "0101" else
                        conv_std_logic_vector(pIDCurrent1Kp, 12)    when rgSwt(3 downto 0) = "0111" else
                        conv_std_logic_vector(pIDCurrent1Ki, 12)    when rgSwt(3 downto 0) = "1000" else
                        conv_std_logic_vector(pIDCurrent1Kd, 12)    when rgSwt(3 downto 0) = "1001" else
                        conv_std_logic_vector(pidCurrent1ICutOffFreq, 12)    when rgSwt(3 downto 0) = "1010" else
                        inVoltage1;

-- Clock for scan voltage generation and seven segment display. Can also be used to limit the input and output
-- sample rates. See comments on AD1a: AD1_controller and DA2a: DA2_controller port maps for more info.

   SampleClock: process (mclk, reset)
   begin
      if reset='1' then 
         cnt1 <= 0;
      elsif rising_edge(mclk) then  -- dividing 50MHz by clockDivider
         if cnt1 = sampleClockDivisor-1 then
            cnt1 <= 0;
         else 
            cnt1 <=cnt1 + 1;
         end if;
         if cnt1 = sampleClockDivisor-1 then
            clksample <= '0';
         elsif cnt1 = sampleClockDivisor/2 then 
            clksample <= '1';
         end if; 
      end if;
   end process;
   
   -- input: 12 bit unsigned to 16 bit signed clamp and rounding functions

   inputSignal <= generatedVoltage1+inputOffset when bypasses(testWaveFormEnableBit)='1' else
                  signed(b"00" & inVoltage1 & b"00")-conv_signed(8192,16)+inputOffset;
   
   -- This process changes multi-byte parameters synchronously, once writing of all the bits is finished by
   -- the attached computer.
   -- This removes any glitches in the computer <-> FPGA communication interface

    changeParameters: process(clkSample)
    variable lastDoneWriting: std_logic := '0';
    
    begin
      if rising_edge(clkSample) then
         if (lastDoneWriting = '0' and DoneWriting = '1' ) then
            inputOffset <= signed(inputOffsetRegister1 & inputOffsetRegister2);
            
            lowPass1CutOffFreq   <= unsigned(cutOffFreq1Register1 & cutOffFreq1Register2 & cutOffFreq1Register3); 
   
            highPass1CutOffFreq  <= unsigned(cutOffFreq2Register1 & cutOffFreq2Register2 & cutOffFreq2Register3);
      
            pidPiezo1Kp <= unsigned(pIDPiezoPRegister1 & pIDPiezoPRegister2 & pIDPiezoPRegister3);
            pidPiezo1Ki <= unsigned(pIDPiezoIRegister1 & pIDPiezoIRegister2 & pIDPiezoIRegister3);
            pidPiezo1Kd <= unsigned(pIDPiezoDRegister1 & pIDPiezoDRegister2 & pIDPiezoDRegister3);
            pidPiezo1Gain <= unsigned(pIDPiezoGRegister1 & pIDPiezoGRegister2 & pIDPiezoGRegister3);
   
            pidCurrent1ICutOffFreq <=  unsigned(pIDCurrentICutOffFreqRegister1 & pIDCurrentICutOffFreqRegister2);
            pidCurrent1Kp <= unsigned(pIDCurrentPRegister1 & pIDCurrentPRegister2 & pIDCurrentPRegister3);
            pidCurrent1Ki <= unsigned(pIDCurrentIRegister1 & pIDCurrentIRegister2& pIDCurrentIRegister3);
            pidCurrent1Kd <= unsigned(pIDCurrentDRegister1 & pIDCurrentDRegister2 & pIDCurrentDRegister3);
            pidCurrent1Gain <= unsigned(pIDCurrentGRegister1 & pIDCurrentGRegister2 & pIDCurrentGRegister3);
  
            setpoint <= signed(pIDPiezoSetPointRegister1 & pIDPiezoSetPointRegister2);
   
            scanGenerator1DeltaY <= unsigned(scanDeltaYRegister1 & scanDeltaYRegister2);
            scanGenerator1DeltaX <= unsigned(scanDeltaXRegister1 & scanDeltaXRegister2);
            scanGenerator1Offset <= signed(scanOffsetRegister1 & scanOffsetRegister2);
            scanGenerator1Amplitude <= unsigned(scanAmplitudeRegister1 & scanAmplitudeRegister2);
         else
            null;
         end if;
         lastDoneWriting := doneWriting;
      end if;
   end process;
   
   -- reset all components by pushing button 2 on the board, or via the computer through USB interface
   reset <= rgBtn(2) or bypasses(resetBit);
   
   -- this bit asserts that the computer is done writing all the bytes of a multi-byte parameter,
   -- asserted by user interface
   doneWriting          <= bypasses2(doneWritingBit); 
   
   lowPass1Bypass       <= bypasses(lowPass1BypassBit);
   lowPass1In           <= inputSignal; -- input goes into filter
      
   highPass1Bypass      <= bypasses(highPass1BypassBit);
   highPass1In          <= lowPass1Out; -- output from low-pass goes into high-pass
   
   scanReset            <= reset OR bypasses2(scanRestartBit); -- reset a scan manually (to restart very slow scans fast)
   
   pidPiezo1Bypass <= bypasses(pIDPiezoBypassBit);
   pidPiezo1In <= highPass1Out;
   pidPiezo1Setpoint <= setPoint when bypasses2(scanSetPointBit) = '0' else
                        generatedVoltage1; -- the setpoint is either a constant, or can be scanned with a triangle wave
   pidPiezo1InitOutput <= scanGenerator1Offset; -- initial PID output is the scan offset
      
   pidCurrent1ICutOffBypass <= bypasses2(pIDCurrentICutOffBypassBit);
   pidCurrent1Polarity <= bypasses2(pIDCurrentPolarityBit);
   pidCurrent1Bypass <= bypasses(pIDCurrentBypassBit);
   pidCurrent1In <= highPass1Out;
   pidCurrent1Setpoint <= setPoint when bypasses2(scanSetPointBit) = '0' else
                        generatedVoltage1; -- the setpoint is either a constant, or can be scanned with a triangle wave
   
   lockSignalPiezo <= pidPiezo1Out;
   lockSignalCurrent <= pidCurrent1Out;
   scanSignal <= generatedVoltage1;
      
   -- 3 states: 
   -- "lock", PID signals are present at the outputs 
   -- "scan", a scan triangle wave is present at output 1, 1.75 V are present at output 2. 
   --    The scan offset serves as the initial piezo PID output, to force the loop to lock to 
   --    the closest lockpoint.
   -- "bypass all", output=input, for testing purposes
      
   combinatorial_module: process(bypasses, present_state, scanSignal, lockSignalPiezo, lockSignalCurrent, inVoltage1, inVoltage2, pidPiezo1Reset)
   begin
      case present_state is
         when scan =>
            outputSignal1 <= scanSignal;
            outputSignal2 <= (OTHERS=>'0');
            pidPiezo1Reset <= '1';
         when lock =>
            outputSignal1 <= lockSignalPiezo;
            outputSignal2 <= lockSignalCurrent;
            pidPiezo1Reset <= reset;
         when bypassall =>
            outputSignal1 <= signed(b"00" & inVoltage1 & b"00")-conv_signed(8192,16);
            outputSignal2 <= signed(b"00" & inVoltage2 & b"00")-conv_signed(8192,16);
            pidPiezo1Reset <= reset;
         when others =>
            null;
      end case;
   end process;
            
   -- selection of current state is based solely on input from user interface
   present_state <= 
      lock when bypasses(lockedBit)='1' else
      bypassall when bypasses2(bypassAllBit)='1' else
      scan;   

   -- output: 16-bit signed to 12-bit unsigned clamp and rounding functions

   -- output1: piezo feedback
   outVoltage1 <= conv_std_logic_vector(0,12) when outputSignal1 < conv_signed(-8192,16) else
      conv_std_logic_vector(4095,12) when outputSignal1 > conv_signed(8191,16) else
      conv_std_logic_vector(outputSignal1+conv_signed(8192,16),16)(13 downto 2);
   
   -- output2: current feedback
   outVoltage2 <= conv_std_logic_vector(0,12) when outputSignal2 < conv_signed(-8192,16) else
      conv_std_logic_vector(4095,12) when outputSignal2 > conv_signed(8191,16) else
      conv_std_logic_vector(outputSignal2+conv_signed(8192,16),16)(13 downto 2);

end lockbox;
 
