----------------------------------------------------------------------------------
-- Company: The University of Oklahoma
-- Engineer: Arne Schwettmann
-- 
-- Create Date:    18:01:58 03/01/2010 
-- Design Name: 
-- Module Name:    lowPass16bit - Behavioral 
-- Project Name:   Shaffer Lab Lockbox
-- Target Devices: Digilent Nexys2 1200k Spartan3 based evaluation board
-- Tool versions: 
-- Description:
--    please see the top level source file "lockbox.vhdl" for release info 
--
-- Dependencies: 
--
-- Revision: 
--    1.0 public release (August 2011)
--
-- Additional Comments: 
--
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity lowPass16bit is
   port (
      clk      : in std_logic;
      cutOffFreqHz   : in unsigned(23 downto 0);
      input    : in signed(15 downto 0);
      output   : out signed(15 downto 0);
      bypass   : in std_logic;
      reset    : in std_logic
   );
end lowPass16bit;

architecture Behavioral of lowPass16bit is

   signal currentOutput : signed(35 downto 0);
begin

   -- implementation of discrete time low pass filter as described e.g. in 
   -- http://en.wikipedia.org/wiki/Low-pass_filter
   -- y_n=alpha*x_(n)+(1-alpha)y_(n-1)
   -- here alpha is the smoothing time constant alpha=DT/(RC+DT)
   -- where RC is related to the cut off frequency in Hertz by
   -- RC=1/(2 pi f). For f much smaller than clk freq, alpha=2 pi f DT
   -- since the effective clkfreq is 50MHz/8, and alpha is 
   -- 6.28*1/clkfreq*f, a multiplication
   -- with alpha is a shift of 20 bits to the right then 
   -- multiplied by the cut off frequency

   -- sequential process calculates filtered signal
   discreteLowPass: process (clk, reset)
   variable intermediateResult1   : signed(35 downto 0);
   variable intermediateResult2   : signed(60 downto 0);
   variable intermediateResult3   : signed(60 downto 0);
   variable intermediateResult4   : signed(60 downto 0);
   variable intermediateResult5   : signed(60 downto 0);
   
   variable lastOutput           : signed(35 downto 0);
   variable currentInput         : signed(35 downto 0);
   
   variable cntr                 : integer := 0;

   begin
      if reset = '1' then
         currentOutput<=conv_signed(0,36);
         cntr := 0;
      else 
         if rising_edge(clk) then
            case cntr is
               when 0 =>
                        lastOutput := currentOutput;
                        if input(input'left)='0' then 
                           currentInput := input & b"0000_0000_0000_0000_0000";
                        else
                           currentInput := input & b"0000_0000_0000_0000_0000";
                        end if;
                        cntr := cntr + 1;
      
               when 1 =>
                  intermediateResult1 := shr(currentInput, conv_unsigned(20,36));
                  cntr := cntr + 1;
               
               when 2 =>
                  intermediateResult2 := intermediateResult1  * cutOffFreqHz;
                  cntr := cntr + 1;
               
               when 3 =>
                  intermediateResult1 := shr(lastOutput, conv_unsigned(20,36));
                  cntr := cntr + 1;
                  
               when 4 =>
                  intermediateResult3 := intermediateResult1 * cutOffFreqHz;
                  cntr := cntr + 1;
               
               when 5 =>
                  intermediateResult3 := lastOutput - intermediateResult3;
                  cntr := cntr + 1;
               
               when 6 =>
                  intermediateResult3 := intermediateResult2 + intermediateResult3;
                  cntr := cntr + 1;
                  
               when 7 =>
                  currentOutput <= intermediateResult3(35 downto 0);
                  cntr := 0;
                  
               when others =>
                  cntr := 0;
                  
               end case;
         end if;
      end if;
   end process;
   
   -- convert to 16 bit and output
   output <= input when bypass = '1' else 
      currentOutput(35 downto 20);
            
   
end Behavioral;