//////////////////////////////////////////////////////////////////////////////////
//
//    PROGRAM LockBoxControlWin
//
// VERSION:
//
// 1.0 (public release 08/2011)
//
// COPYRIGHT/AUTHORS:
//
// This program is public domain. However, he who makes use of this
// program should give credit to the authors and cite the reference:
//
// Arne Schwettmann, Jonathon Sedlacek, and James P. Shaffer,
// "FPGA-based locking circuit for external cavity diode laser frequency
// stabilization," Rev. Sci. Inst. (2011)
//
// We will try to give limited support, but can not make any guarantees
// contact: shaffer@nhn.ou.edu
//
// PORTABILITY:
//
// Written for Microsoft Windows using Borland Developer Studio 2006 - C++ Builder.
//
// This program makes use of the VCL libraries.
//
// It needs the Digilent ADEPT Suite to be installed which result in the needed
// file "dpcutil.dll" to be present in the windows/system32 directory.
//
// The "dpcutilOMF.lib" file needs to be included in the project.
// dpcutilOMF.lib is a version of the Digilent USB communication library interface
// dpcutil.lib, converted with the Borland tool coff2omf to allow it to link
// against the Borland C++ compiler.
//
// DESCRIPTION:
//
// This is the graphical user interface (GUI) for the Shaffer Lab laser
// locking circuit FPGA implementation. It communicates via USB with an attached
// Digilent Nexys2 FPGA board that runs the Shaffer Lab Lockbox locking circuit
// program, which is written in VHDL. This GUI allows the user to set
// locking circuit parameters such as proportional, integral, and differential
// gains, filter cut-off frequencies, the scan range and offset. It also allows
// the user to reset the fpga, and change between "scan," "locked," and "debug"
// modes. The menu provides options to select the FPGA to connect to, and to
// save and or load configuration files. A config file stores all parameters
// and allows to easily go back to previous settings. A file names "default.cfg"
// is automatically saved on program exit, and loaded on program startup.
//
// The easiest and fastest way to change parameters is to use the cursor keys:
// Activate an edit field with the mouse, and then use cursor left right to select
// the digit and cursor up/down to change the parameter by that order of magnitude.
// Parameters can also be changed by the mouse, by first positioning the cursor
// over the digit one wants to change, and then clicking the plus/minus buttons
// to the left of the respective edit field.
//
// Mouse-over hints can be enabled from the Help menu. If enabled, a little help
// text is displayed when hovering the mouse over a GUI element.
//
// Written for:
//    Dr. James P. Shaffer's research group,
//    Homer L. Dodge Dept. of Physics
//    University of Oklahoma
//    e-mail: shaffer@nhn.ou.edu
//    More contact info can be found at www.nhn.ou.edu
//
// Revision:
//    1.0 public release (August 2011)
//
// Additional Comments:
//
// USES: dpcutil
//
// NOTE:
// Digilent ADEPT Suite needs to be installed on the system
// - this results in the needed dpcutil.dll to be present in windows/system32.
//
// PROJECT FILES:
// LockBoxControlWin.cpp - IDE managed program entry point
// LockBoxControlWin.res - IDE managed program resources
// dpcutilOMF.lib - Interface to DPCUTIL.DLL, converted to OMF format
// LockBoxControlWinMain.cpp - main program source
// LockBoxControlWinMain.dfm - form definition (contains VCL user interface layout such as button positions)
// LockBoxControlWinMain.h - declarations
// gendefs.h - Global Definitions for DPCDEMO Program
// dpcutil.h - Interface Declarations for DPCUTIL.DLL
// dpcdefs.h - Type and constant definitions for DPCUTIL.DLL
//
// THINGS TO DO:
//
//////////////////////////////////////////////////////////////////////////////////




//---------------------------------------------------------------------------

#include <vcl.h>
#pragma hdrstop

#include "LockBoxControlWinMain.h"
#include "math.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"

#include <windows.h>
#include <stdio.h>

//dpcdemo includes
#include "gendefs.h" /* holds definitions for the DPCDEMO program */

//dpcutil includes
#include "dpcdefs.h"	/* holds error codes and data types for dpcutil	*/
#include "dpcutil.h"	/* holds declaration of DPCUTIL API calls		*/

TForm3 *Form3;
//---------------------------------------------------------------------------
//Initialization on program startup
__fastcall TForm3::TForm3(TComponent* Owner)
  : TForm(Owner)
{
  //get current directory and create a default config filename
  programDirectory=GetCurrentDir();
  defaultConfigFileName=programDirectory+AnsiString("\\default.cfg");

  //make the mouseover hint delay longer (for "help/enable hints")
  Application->HintHidePause=10000;

  //initialize USB communication
  FInit();

  //default piezo PID parameters
  pIDPiezo.bypass=0;
  pIDPiezo.kP=1;
  pIDPiezo.kI=1;
  pIDPiezo.kD=1;
  pIDPiezo.g=100;
  pIDPiezo.setPoint=0;

  //default current PID parameters
  pIDCurrent.bypass=1;
  pIDCurrent.kP=1;
  pIDCurrent.kI=1;
  pIDCurrent.kD=1;
  pIDCurrent.g=100;
  pIDCurrent.setPoint=0;
  pIDCurrent.iCutOffFreq=100;
  pIDCurrent.cutOffBypass=false;
  pIDCurrent.polarity=false;

  //for debugging only
  //(allows to generate a test triangle wave to use as fake input to the PIDs)
  testWave.enable=1;
  testWave.nr=0;

  //default low pass filter parameters
  lowPass1.cutOffFreq=500;
  lowPass1.bypass=1;

  //default high pass filter parameters
  highPass1.cutOffFreq=500;
  highPass1.bypass=1;

  //default input offset and gain parameters
  input1.offset=0;
  input1.gain=1;

  //default scan waveform parameters
  scan1.bypass=false;
  scan1.amplitude=32767;
  scan1.frequency=50;
  scan1.offset=0;

  //start in "scan" mode
  lock=false;
  bypassAll=false;
  scanSetPoint=false;

  //load last parameters from the config file, if it exists
  //and update interface and FPGS with the current parameters
  LoadSettings(defaultConfigFileName);
}


/* These functions are taken from Digilent Adept Suite SDK
   DPCUTIL Demo

/* ------------------------------------------------------------ */

/***	FInit
**
**	Description:
**		Initialize application local variables.
**
*/

BOOL TForm3::FInit() {

	ERC erc;
	int idDvc;

	/* DPCUTIL API CALL : DpcInit
	*/
	if (!DpcInit(&erc)) {
		return fFalse;
	}

	/* DPCUTIL API CALL : DvmgGetDefaultDev
	*/
	idDvc = DvmgGetDefaultDev(&erc);
	if (idDvc == -1) {
		//printf("No default device\n");
		return fTrue;
	}
	else {
		/* DPCUTIL API CALL : DvmgGetDevName
		*/
		DvmgGetDevName(idDvc, szDefDvc, &erc);
		strcpy(szDefDvc, szDefDvc);
		return fTrue;
	}
}

/* ------------------------------------------------------------ */

/***	DoDvcTbl
**
**	Description:
**		Makes API call to open Communication Devices dialog box
**
*/

void TForm3::DoDvcTbl() {

	ERC erc;
	int idDvc;

	//printf("Opening Communication Modules dialog box...\n");

	/* DPCUTIL API CALL : DvmgStartConfigureDevices
	*/
	DvmgStartConfigureDevices(NULL, &erc);

	/* DPCUTIL API CALL : DvmgGetDefaultDev
	*/
	idDvc = DvmgGetDefaultDev(&erc);
	if (idDvc == -1) {
		//printf("No default device selected\n");
	}
	else {

		/* DPCUTIL API CALL : DvmgGetDevName
		*/
		DvmgGetDevName(idDvc, szDefDvc, &erc);
		//printf("Default device selected:  %s\n", szDefDvc);
	}
}

/* ------------------------------------------------------------ */

/***	DoGetReg
**
**	Description:
**		Gets a byte from, a specified register
**
*/

void TForm3::DoGetReg(BYTE idReg, BYTE &idData) {

	ERC		erc;
	HANDLE	hif;

	/* DPCUTIL API CALL : DpcOpenData
	*	notice last parameter is set to NULL.  This is a blocking call
	*/
	if (!DpcOpenData(&hif, szDefDvc, &erc, NULL)) {
		//printf("DpcOpenData failed.\n");
		goto lFail;
	}

	/* DPCUTIL API CALL : DpcGetReg
	*/
	if (!DpcGetReg(hif, idReg, &idData, &erc, NULL)) {
		DpcCloseData(hif,&erc);
		//printf("DpcGetReg failed.\n");
		goto lFail;
	}

	/* DPCUTIL API CALL : DpcGetFirstError
	*/
	erc = DpcGetFirstError(hif);

	if (erc == ercNoError) {
		/* DPCUTIL API CALL : DpcCloseData
		*/
		DpcCloseData(hif, &erc);
		//printf("Complete!  Data received = %d\n", idData);
	}
	else{
		/* DPCUTIL API CALL : DpcCloseData
		*/
		DpcCloseData(hif, &erc);
		//printf("An error occured while reading the register.\n", idData);
	}

lFail:
	return;

}

/* ------------------------------------------------------------ */

/***	DoPutReg
**
**	Description:
**		Sends a byte to a specified register
**
*/

void TForm3::DoPutReg(BYTE	idReg, BYTE	idData) {

	ERC		erc;
	HANDLE	hif;
	TRID	trid;

		/* DPCUTIL API CALL : DpcOpenData
	*	notice the last parameter is a TRID address.  This is a non-blocking call
	*/
	if (!DpcOpenData(&hif, szDefDvc, &erc, &trid)) {
		//printf("DpcOpenData failed.\n");
		goto lFail;
	}

	/* we must wait for the transaction to be completed before moving on */
	if (!DpcWaitForTransaction(hif, trid, &erc)) {
		DpcCloseData(hif, &erc);
		//printf("DpcOpenData failed.\n");
		goto lFail;
	}

	/* DPCUTIL API CALL : DpcGetReg
	*/
	if (!DpcPutReg(hif, idReg, idData, &erc, NULL)) {
		DpcCloseData(hif,&erc);
		//printf("DpcGetReg failed.\n");
		goto lFail;
	}

	/* DPCUTIL API CALL : DpcGetFirstError
	*/
	erc = DpcGetFirstError(hif);

	if (erc == ercNoError) {
		/* DPCUTIL API CALL : DpcCloseData
		*/
		DpcCloseData(hif, &erc);
		//printf("Complete!  Register set.\n");
	}
	else{
		/* DPCUTIL API CALL : DpcCloseData
		*/
		DpcCloseData(hif, &erc);
		//printf("An error occured while setting the register.\n", idData);
	}

	lFail:
		return;
}
/* end of functions taken from DPCUTIL Demo */

//---------------------------------------------------------------------------

/*
  DoneWriting routine
  writing a 24 (16) bit number to the FPGA via DPCUTIL requires writing 3
  (2) 8 bit registers sequentially. In order to keep the PID loops from
  glitching when changing a parameter value, the parameters in the FPGA
  are updated only after all 24 (16) bits have been transferred. The FPGA
  is notified by setting the doneWriting bit high for longer than a 1 us.
*/

void TForm3::DoneWriting()
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses2,oldBypasses);

  DoPutReg(reg_bypasses2, oldBypasses | 1<<reg_doneWritingBit);
  Sleep(10);
  DoPutReg(reg_bypasses2,oldBypasses & ~(1<<reg_doneWritingBit));
  Sleep(10);
}
//---------------------------------------------------------------------------

//Save settings routine
//writes the settings to a binary .cfg file

void TForm3::SaveSettings(AnsiString fileName)
{
	bool dialogSuccess = true;
	if ( fileName==NULL )
	{
		dialogSuccess = SaveDialog1->Execute();
		fileName = SaveDialog1->FileName;
	}

	if (dialogSuccess) {
		TFileStream *FStream = new TFileStream(fileName.c_str(),fmCreate);
		int memsize = sizeof(pIDPiezo);
		FStream->Write(&pIDPiezo,memsize);
		memsize = sizeof(pIDCurrent);
		FStream->Write(&pIDCurrent,memsize);
		memsize = sizeof(lowPass1);
		FStream->Write(&lowPass1,memsize);
    memsize = sizeof(highPass1);
		FStream->Write(&highPass1,memsize);
    memsize = sizeof(input1);
		FStream->Write(&input1,memsize);
    memsize = sizeof(scan1);
		FStream->Write(&scan1,memsize);
    memsize = sizeof(lock);
		FStream->Write(&lock,memsize);
    memsize = sizeof(testWave);
    FStream->Write(&testWave,memsize);
    memsize = sizeof(bypassAll);
    FStream->Write(&bypassAll,memsize);
    memsize = sizeof(scanSetPoint);
    FStream->Write(&scanSetPoint,memsize);
		delete FStream;
	}
}
//---------------------------------------------------------------------------

//loads the settings from a specified .cfg filename
//Load settings routine

void TForm3::LoadSettings(AnsiString fileName)
{
	bool dialogSuccess = true;
	if ( fileName==NULL )
	{
		dialogSuccess = OpenDialog1->Execute();
		fileName = OpenDialog1->FileName;
	}

	if (dialogSuccess) {
    bool openError=false;
    TFileStream *FStream;
    try {
      	FStream = new TFileStream(fileName.c_str(),fmOpenRead);

    } catch (EFOpenError *err) {
		  openError=true;
	  };
	  if (!openError) {
		  int memsize;
      // loads config data from file
      memsize = sizeof(pIDPiezo);
      FStream->Read(&pIDPiezo,memsize);
      memsize = sizeof(pIDCurrent);
      FStream->Read(&pIDCurrent,memsize);
      memsize = sizeof(lowPass1);
      FStream->Read(&lowPass1,memsize);
      memsize = sizeof(highPass1);
      FStream->Read(&highPass1,memsize);
      memsize = sizeof(input1);
      FStream->Read(&input1,memsize);
      memsize = sizeof(scan1);
      FStream->Read(&scan1,memsize);
      memsize = sizeof(lock);
      FStream->Read(&lock,memsize);
      memsize = sizeof(testWave);
      FStream->Read(&testWave,memsize);
      memsize = sizeof(bypassAll);
      FStream->Read(&bypassAll,memsize);
      memsize = sizeof(scanSetPoint);
      FStream->Read(&scanSetPoint,memsize);
    }
    delete FStream;
    //update interface and FPGA
    UpdateGUIAndFPGA();
  }
}
//---------------------------------------------------------------------------

//Updates GUI and FPGA with current parameters,
//for example after config file load

void TForm3::UpdateGUIAndFPGA()
{
  PIDPiezoPEdit->Text=IntToStr(pIDPiezo.kP);
  PIDPiezoPEditExit(this);
  PIDPiezoIEdit->Text=IntToStr(pIDPiezo.kI);
  PIDPiezoIEditExit(this);
  PIDPiezoDEdit->Text=IntToStr(pIDPiezo.kD);
  PIDPiezoDEditExit(this);
  PIDPiezoGEdit->Text=IntToStr(pIDPiezo.g);
  PIDPiezoGEditExit(this);
  PIDPiezoSetPointEdit->Text=IntToStr(pIDPiezo.setPoint);
  PIDPiezoSetPointEditExit(this);
  PIDPiezoEnableCheckBox->Checked=!pIDPiezo.bypass;
  PIDPiezoEnableCheckBoxClick(this);

  PIDCurrentPEdit->Text=IntToStr(pIDCurrent.kP);
  PIDCurrentPEditExit(this);
  PIDCurrentIEdit->Text=IntToStr(pIDCurrent.kI);
  PIDCurrentIEditExit(this);
  PIDCurrentDEdit->Text=IntToStr(pIDCurrent.kD);
  PIDCurrentDEditExit(this);
  PIDCurrentGEdit->Text=IntToStr(pIDCurrent.g);
  PIDCurrentGEditExit(this);
  PIDCurrentEnableCheckBox->Checked=!pIDCurrent.bypass;
  PIDCurrentEnableCheckBoxClick(this);
  PIDCurrentICutOffFrequencyEdit->Text=IntToStr(pIDCurrent.iCutOffFreq);
  PIDCurrentICutOffFrequencyEditExit(this);
  PIDCurrentICutOffEnableCheckBox->Checked=!pIDCurrent.cutOffBypass;
  PIDCurrentICutOffEnableCheckBoxClick(this);
  PIDCurrentPolarityCheckBox->Checked=pIDCurrent.polarity;
  PIDCurrentPolarityCheckBoxClick(this);

  LockCheckBox->Checked=lock;
  LockCheckBoxClick(this);

  ScanFrequencyEdit->Text=IntToStr(scan1.frequency);
  ScanFrequencyEditExit(this);
  ScanAmplitudeEdit->Text=IntToStr(scan1.amplitude);
  ScanAmplitudeEditExit(this);
  ScanOffsetEdit->Text=IntToStr(scan1.offset);
  ScanOffsetEditExit(this);
  ScanEnableCheckBox->Checked=!scan1.bypass;
  ScanEnableCheckBoxClick(this);

  InputOffsetEdit->Text=IntToStr(input1.offset);
  InputOffsetEditExit(this);
  InputGainEdit->Text=IntToStr(input1.gain);
  InputGainEditExit(this);

  InputLowPassCutOffFreqEdit->Text=IntToStr(lowPass1.cutOffFreq);
  InputLowPassCutOffFreqEditExit(this);
  InputLowPassEnableCheckBox->Checked=!lowPass1.bypass;
  InputLowPassEnableCheckBoxClick(this);

  InputHighPassCutOffFreqEdit->Text=IntToStr(highPass1.cutOffFreq);
  InputHighPassCutOffFreqEditExit(this);
  InputHighPassEnableCheckBox->Checked=!highPass1.bypass;
  InputHighPassEnableCheckBoxClick(this);

  TestWaveformEnableCheckBox->Checked=testWave.enable;
  TestWaveformEnableCheckBoxClick(this);
  TestWaveformRadioGroup->ItemIndex=testWave.nr;
  TestWaveformRadioGroupClick(this);

  BypassAllCheckBox->Checked=bypassAll;
  BypassAllCheckBoxClick(this);

  ScanSetPointCheckBox->Checked=scanSetPoint;
  ScanSetPointCheckBoxClick(this);

}
//---------------------------------------------------------------------------

int __fastcall TForm3::Signum(int a)
{
  if (a>0)
    return +1;
  else if (a<0)
    return -1;
  else
    return 0;
}
//---------------------------------------------------------------------------

//save settings on close
void __fastcall TForm3::FormClose(TObject *Sender, TCloseAction &Action)
{
	SaveSettings(defaultConfigFileName);
}
//---------------------------------------------------------------------------


//enable tool tip hints from menu click

void __fastcall TForm3::EnableHints1Click(TObject *Sender)
{
	if (EnableHints1->Checked==false) {
		ShowHint=true;
		EnableHints1->Checked=true;
	}
	else {
		ShowHint=false;
		EnableHints1->Checked=false;
	}
}
//---------------------------------------------------------------------------

//increase a parameter value by an amount depending on the position
//of the cursor in the edit field. If the user clicks the plus button or
//presses the cursor up key, the value of the currently selected parameter
//will be increased. The amount of increase depends on the cursor position
//in the edit field. If the cursor is on the ones, the parameter will be inc.
//by one. If the cursor is on the tens, the parameter will be incr. by ten,
//and so on. This allows for an easy to use parameter change over many
//orders of magnitudes.
void __fastcall TForm3::IncreaseEditValue(TEdit *edit, int maxValue, int minValue, int oldValue)
{
  int currentValue=GetEditIntValue(edit, maxValue, minValue, oldValue);
  int numDigits=0;
  if (currentValue!=0)
    numDigits=(int) (log((double) abs(currentValue))/log((double) 10));
  int selStart=edit->SelStart;
  int selLength=edit->SelLength;
  int digitToChange;
  if (currentValue<0) {
    if ((selStart==0 && (selLength==0 || selLength==1)) ||
        (selStart==1 && selLength==0)) {
      digitToChange=numDigits+1;
    }
    else
      digitToChange=numDigits-selStart-selLength+2;
  }
  else
    digitToChange=numDigits-selStart-selLength+1;
  int newValue=currentValue+pow(10,digitToChange);
  if (Signum(newValue)*Signum(currentValue)<0) {
    newValue=(-1)*currentValue+pow(10,digitToChange);
  }
  newValue=SetEditIntValue(edit, maxValue, minValue, newValue);
  int newNumDigits=0;
  if (newValue!=0)
    newNumDigits=(int) (log((double) abs(newValue))/log((double) 10));
  edit->AutoSelect=false;
  edit->SetFocus();
  PageControl1->SetFocus();
  edit->SetFocus();
  edit->AutoSelect=true;
  if (newValue<0) {
    if (newNumDigits<numDigits) {
      if (selStart!=0) {
        selStart=selStart-1;
      }
    }
    if (selLength==0) {
      if (selStart==0 || selStart==1 )
        selStart=1;
      else
        selStart=selStart-1;
      selLength=1;
    }
    if (selStart==0) {
      selStart=1;
      if (selLength>1)
        selLength=selLength-1;
    }
  }
  else {
    if (newNumDigits>numDigits && currentValue>=0) {
        selStart=selStart+1;
    }
    if (selLength==0) {
      if (selStart!=0)
        selStart=selStart-1;
      selLength=1;
    }
  }
  if (abs(newValue)<pow(10,digitToChange) && digitToChange>0) {
    selStart=0;
    selLength=0;
  }
  edit->SelStart=selStart;
  edit->SelLength=selLength;
};
//---------------------------------------------------------------------------

//decrease a parameter value by an amount depending on the position
//of the cursor in the edit field. If the user clicks the minus button or
//presses the cursor down key, the value of the currently selected parameter
//will be increased. The amount of increase depends on the cursor position
//in the edit field. If the cursor is on the ones, the parameter will be inc.
//by one. If the cursor is on the tens, the parameter will be incr. by ten,
//and so on. This allows for an easy to use parameter change over many
//orders of magnitudes.
void __fastcall TForm3::DecreaseEditValue(TEdit *edit, int maxValue, int minValue, int oldValue)
{
  int currentValue=GetEditIntValue(edit, maxValue, minValue, oldValue);
  int numDigits=0;
  if (currentValue!=0)
    numDigits=(int) (log((double) abs(currentValue))/log((double) 10));
  int selStart=edit->SelStart;
  int selLength=edit->SelLength;
  int digitToChange;
  if (currentValue<0) {
    if ((selStart==0 && (selLength==0 || selLength==1)) ||
        (selStart==1 && selLength==0)) {
      digitToChange=numDigits+1;
    }
    else
      digitToChange=numDigits-selStart-selLength+2;
  }
  else
    digitToChange=numDigits-selStart-selLength+1;
  int newValue=currentValue-pow(10,digitToChange);
  if (Signum(newValue)*Signum(currentValue)<0)
    newValue=(-1)*currentValue-pow(10,digitToChange);
  newValue=SetEditIntValue(edit, maxValue, minValue, newValue);
  int newNumDigits=0;
  if (newValue!=0)
    newNumDigits=(int) (log((double) abs(newValue))/log((double) 10));
  edit->AutoSelect=false;
  edit->SetFocus();
  PageControl1->SetFocus();
  edit->SetFocus();
  edit->AutoSelect=true;

  if (newValue<0) {
    if (newNumDigits>numDigits) {
      selStart=selStart+1;
    }
    if (selLength==0) {
      if (selStart==0 || selStart==1 )
        selStart=1;
      else
        selStart=selStart-1;
      selLength=1;
    }
    if (selStart==0) {
      selStart=1;
      if (selLength>1)
        selLength=selLength-1;
    }
  }
  else {
    if (newNumDigits<numDigits) {
      if (selStart!=0) {
        selStart=selStart-1;
      }
    }
    if (selLength==0) {
      if (selStart!=0)
        selStart=selStart-1;
      selLength=1;
    }
  }
  if (abs(newValue)<pow(10,digitToChange) && digitToChange>0) {
    selStart=0;
    selLength=0;
  }
  edit->SelStart=selStart;
  edit->SelLength=selLength;
};
//---------------------------------------------------------------------------

//read a value from a specified edit field in the front end
int __fastcall TForm3::GetEditIntValue(TEdit *edit, int maxValue, int minValue, int oldValue)
{
  int newValue;
  bool conversionError=false;
  try {
		newValue=edit->Text.ToInt();
	} catch (EConvertError *err) {
		conversionError=true;
	};
	if (conversionError
		|| newValue>maxValue
		|| newValue<minValue) {
		edit->Text=IntToStr(oldValue);
		conversionError=false;
		return oldValue;
	}
  else {
    return newValue;
  };
};
//---------------------------------------------------------------------------

//set a value in a specified edit field in the front end
int __fastcall TForm3::SetEditIntValue(TEdit *edit, int maxValue, int minValue, int newValue)
{
  if (newValue<=maxValue
		&& newValue>=minValue) {
		edit->Text=IntToStr(newValue);
    return newValue;
  }
  else
    return StrToInt(edit->Text);
};
//---------------------------------------------------------------------------

//write the input offset value to the FPGA
void __fastcall TForm3::InputOffsetEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(InputOffsetEdit, 32767, -32768, input1.offset);
  BYTE highBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_inputOffsetRegister1,highBits);
  DoPutReg(reg_inputOffsetRegister2,lowBits);
  DoneWriting();

  input1.offset=newValue;
}
//---------------------------------------------------------------------------

//click on input offset + button
void __fastcall TForm3::InputOffsetIncreaseButttonClick(TObject *Sender)
{
    IncreaseEditValue(InputOffsetEdit, 32767, -32768, input1.offset);
}
//---------------------------------------------------------------------------

//click on input offset - button
void __fastcall TForm3::InputOffsetDecreaseButttonClick(TObject *Sender)
{
    DecreaseEditValue(InputOffsetEdit, 32767, -32768, input1.offset);
}
//---------------------------------------------------------------------------

//cursor up/down key press in input offset edit field
void __fastcall TForm3::InputOffsetEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          InputOffsetIncreaseButttonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          InputOffsetDecreaseButttonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//write the input low pass cut off freq value to the FPGA
void __fastcall TForm3::InputLowPassCutOffFreqEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(InputLowPassCutOffFreqEdit, 16777215, 0, lowPass1.cutOffFreq);

  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_cutOffFreq1Register1,highBits);
  DoPutReg(reg_cutOffFreq1Register2,mediumBits);
  DoPutReg(reg_cutOffFreq1Register3,lowBits);
  DoneWriting();

  lowPass1.cutOffFreq=newValue;
}
//---------------------------------------------------------------------------

//click on input low pass cut off frequency + button
void __fastcall TForm3::InputLowPassCutOffFreqIncreaseButtonClick(
      TObject *Sender)
{
  IncreaseEditValue(InputLowPassCutOffFreqEdit, 16777215, 0, lowPass1.cutOffFreq);
}
//---------------------------------------------------------------------------

//click on input low pass cut off frequency - button
void __fastcall TForm3::InputLowPassCutOffFreqDecreaseButtonClick(
      TObject *Sender)
{
  DecreaseEditValue(InputLowPassCutOffFreqEdit, 16777215, 0, lowPass1.cutOffFreq);
}
//---------------------------------------------------------------------------

//cursor up/down key press in input offset edit field
void __fastcall TForm3::InputLowPassCutOffFreqEditKeyDown(TObject *Sender,
      WORD &Key, TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          InputLowPassCutOffFreqIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          InputLowPassCutOffFreqDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//write the test waveform enable bit to FPGA
void __fastcall TForm3::TestWaveformEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (TestWaveformEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_testWaveFormEnableBit);
    testWave.enable=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_testWaveFormEnableBit));
    testWave.enable=false;
  }
}
//---------------------------------------------------------------------------

//respond to click on test wave form buttons (for debugging only)
void __fastcall TForm3::TestWaveformRadioGroupClick(TObject *Sender)
{
  if (TestWaveformEnableCheckBox->Checked) {
    //DoPutReg(reg_testWaveformNr,TestWaveformRadioGroup->ItemIndex);
    testWave.nr=TestWaveformRadioGroup->ItemIndex;
  }
}
//---------------------------------------------------------------------------

//*** Piezo PID Controls

//write the piezo PID overall gain setting to the fpga
void __fastcall TForm3::PIDPiezoGEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDPiezoGEdit, 16777215, 0, pIDPiezo.g);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDPiezoGRegister1,highBits);
  DoPutReg(reg_pIDPiezoGRegister2,mediumBits);
  DoPutReg(reg_pIDPiezoGRegister3,lowBits);
  DoneWriting();

  pIDPiezo.g=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on piezo PID overall gain edit field
void __fastcall TForm3::PIDPiezoGEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDPiezoGIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDPiezoGDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on piezo PID overall gain + button
void __fastcall TForm3::PIDPiezoGIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDPiezoGEdit, 16777215, 0, pIDPiezo.g);
}
//---------------------------------------------------------------------------

//click on piezo PID overall gain + button
void __fastcall TForm3::PIDPiezoGDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDPiezoGEdit, 16777215, 0, pIDPiezo.g);
}
//---------------------------------------------------------------------------

//write the piezo PID proportional gain setting to the fpga
void __fastcall TForm3::PIDPiezoPEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDPiezoPEdit, 16777215, 0, pIDPiezo.kP);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDPiezoPRegister1,highBits);
  DoPutReg(reg_pIDPiezoPRegister2,mediumBits);
  DoPutReg(reg_pIDPiezoPRegister3,lowBits);
  DoneWriting();

  pIDPiezo.kP=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on piezo PID proportional gain edit field
void __fastcall TForm3::PIDPiezoPEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDPiezoPIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDPiezoPDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on piezo PID proportional gain + button
void __fastcall TForm3::PIDPiezoPIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDPiezoPEdit, 16777215, 0, pIDPiezo.kP);
}
//---------------------------------------------------------------------------

//click on piezo PID proportional gain - button
void __fastcall TForm3::PIDPiezoPDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDPiezoPEdit, 16777215, 0, pIDPiezo.kP);
}
//---------------------------------------------------------------------------

//write the piezo PID integral gain setting to the fpga
void __fastcall TForm3::PIDPiezoIEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDPiezoIEdit, 16777215, 0, pIDPiezo.kI);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDPiezoIRegister1,highBits);
  DoPutReg(reg_pIDPiezoIRegister2,mediumBits);
  DoPutReg(reg_pIDPiezoIRegister3,lowBits);
  DoneWriting();

  pIDPiezo.kI=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on piezo PID integral gain edit field
void __fastcall TForm3::PIDPiezoIEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDPiezoIIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDPiezoIDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on piezo PID integral gain + button
void __fastcall TForm3::PIDPiezoIIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDPiezoIEdit, 16777215, 0, pIDPiezo.kI);
}
//---------------------------------------------------------------------------

//click on piezo PID integral gain - button
void __fastcall TForm3::PIDPiezoIDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDPiezoIEdit, 16777215, 0, pIDPiezo.kI);
}
//---------------------------------------------------------------------------

//write the piezo PID differential gain setting to the fpga
void __fastcall TForm3::PIDPiezoDEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDPiezoDEdit, 16777215, 0, pIDPiezo.kD);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDPiezoDRegister1,highBits);
  DoPutReg(reg_pIDPiezoDRegister2,mediumBits);
  DoPutReg(reg_pIDPiezoDRegister3,lowBits);
  DoneWriting();

  pIDPiezo.kD=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on piezo PID differential gain edit field
void __fastcall TForm3::PIDPiezoDEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDPiezoDIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDPiezoDDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on piezo PID differential gain + button
void __fastcall TForm3::PIDPiezoDIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDPiezoDEdit, 16777215, 0, pIDPiezo.kD);
}
//---------------------------------------------------------------------------

//click on piezo PID differential gain - button
void __fastcall TForm3::PIDPiezoDDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDPiezoDEdit, 16777215, 0, pIDPiezo.kD);
}
//---------------------------------------------------------------------------

//write the piezo PID enable bit to the fpga
void __fastcall TForm3::PIDPiezoEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (!PIDPiezoEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_pIDPiezoBypassBit);

    pIDPiezo.bypass=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_pIDPiezoBypassBit));

    pIDPiezo.bypass=false;
  }
}
//---------------------------------------------------------------------------

//*** Current PID Controls

//write the current PID overall gain setting to the fpga
void __fastcall TForm3::PIDCurrentGEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDCurrentGEdit, 16777215, 0, pIDCurrent.g);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDCurrentGRegister1,highBits);
  DoPutReg(reg_pIDCurrentGRegister2,mediumBits);
  DoPutReg(reg_pIDCurrentGRegister3,lowBits);
  DoneWriting();

  pIDCurrent.g=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on current PID overall gain edit field
void __fastcall TForm3::PIDCurrentGEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDCurrentGIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDCurrentGDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on current PID overall gain + button
void __fastcall TForm3::PIDCurrentGIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDCurrentGEdit, 16777215, 0, pIDCurrent.g);
}
//---------------------------------------------------------------------------

//click on current PID overall gain - button
void __fastcall TForm3::PIDCurrentGDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDCurrentGEdit, 16777215, 0, pIDCurrent.g);
}
//---------------------------------------------------------------------------

//write the current PID proportional gain setting to the fpga
void __fastcall TForm3::PIDCurrentPEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDCurrentPEdit, 16777215, 0, pIDCurrent.kP);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDCurrentPRegister1,highBits);
  DoPutReg(reg_pIDCurrentPRegister2,mediumBits);
  DoPutReg(reg_pIDCurrentPRegister3,lowBits);
  DoneWriting();

  pIDCurrent.kP=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on current PID proportional gain edit field
void __fastcall TForm3::PIDCurrentPEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDCurrentPIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDCurrentPDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on current PID proportional gain + button
void __fastcall TForm3::PIDCurrentPIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDCurrentPEdit, 16777215, 0, pIDCurrent.kP);
}
//---------------------------------------------------------------------------

//click on current PID proportional gain - button
void __fastcall TForm3::PIDCurrentPDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDCurrentPEdit, 16777215, 0, pIDCurrent.kP);
}
//---------------------------------------------------------------------------

//write the current PID integral gain setting to the fpga
void __fastcall TForm3::PIDCurrentIEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDCurrentIEdit, 16777215, 0, pIDCurrent.kI);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDCurrentIRegister1,highBits);
  DoPutReg(reg_pIDCurrentIRegister2,mediumBits);
  DoPutReg(reg_pIDCurrentIRegister3,lowBits);
  DoneWriting();

  pIDCurrent.kI=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on currentPID integral gain edit field
void __fastcall TForm3::PIDCurrentIEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDCurrentIIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDCurrentIDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on current PID integral gain + button
void __fastcall TForm3::PIDCurrentIIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDCurrentIEdit, 16777215, 0, pIDCurrent.kI);
}
//---------------------------------------------------------------------------

//click on current PID integral gain - button
void __fastcall TForm3::PIDCurrentIDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDCurrentIEdit, 16777215, 0, pIDCurrent.kI);
}
//---------------------------------------------------------------------------

//write the current PID differential gain setting to the fpga
void __fastcall TForm3::PIDCurrentDEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDCurrentDEdit, 16777215, 0, pIDCurrent.kD);
  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDCurrentDRegister1,highBits);
  DoPutReg(reg_pIDCurrentDRegister2,mediumBits);
  DoPutReg(reg_pIDCurrentDRegister3,lowBits);
  DoneWriting();

  pIDCurrent.kD=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on current PID differential gain edit field
void __fastcall TForm3::PIDCurrentDEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDCurrentDIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDCurrentDDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on current PID differential gain + button
void __fastcall TForm3::PIDCurrentDIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDCurrentDEdit, 16777215, 0, pIDCurrent.kD);
}
//---------------------------------------------------------------------------

//click on current PID differential gain - button
void __fastcall TForm3::PIDCurrentDDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDCurrentDEdit, 16777215, 0, pIDCurrent.kD);
}
//---------------------------------------------------------------------------

//write the current PID enable bit to the fpga
void __fastcall TForm3::PIDCurrentEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (!PIDCurrentEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_pIDCurrentBypassBit);

    pIDCurrent.bypass=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_pIDCurrentBypassBit));

    pIDCurrent.bypass=false;
  }
}
//---------------------------------------------------------------------------

//write the current PID polarity bit to the fpga (180 degree phaseshift on
//input)
void __fastcall TForm3::PIDCurrentPolarityCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses2,oldBypasses);

  if (PIDCurrentPolarityCheckBox->Checked) {
    DoPutReg(reg_bypasses2, oldBypasses | 1<<reg_pIDCurrentPolarityBit);

    pIDCurrent.polarity=true;
  }
  else {
    DoPutReg(reg_bypasses2,oldBypasses & ~(1<<reg_pIDCurrentPolarityBit));

    pIDCurrent.polarity=false;
  }
}
//---------------------------------------------------------------------------

//write the current PID cut off frequency bypass bit to the fpga
void __fastcall TForm3::PIDCurrentICutOffEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses2,oldBypasses);

  if (!PIDCurrentICutOffEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses2, oldBypasses | 1<<reg_PIDCurrentICutOffBypassBit);

    pIDCurrent.cutOffBypass=true;
  }
  else {
    DoPutReg(reg_bypasses2,oldBypasses & ~(1<<reg_PIDCurrentICutOffBypassBit));

    pIDCurrent.cutOffBypass=false;
  }
}
//---------------------------------------------------------------------------

//write the current PID cut off frequency setting to the fpga
//this is a high pass: the current PID acts only on non-dc frequencies
//to prevent the current and piezo PID from working against each other
void __fastcall TForm3::PIDCurrentICutOffFrequencyEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDCurrentICutOffFrequencyEdit, 65535, 0, pIDCurrent.iCutOffFreq);
  BYTE highBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDCurrentICutOffFreqRegister1,highBits);
  DoPutReg(reg_pIDCurrentICutOffFreqRegister2,lowBits);
  DoneWriting();

  pIDCurrent.iCutOffFreq=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on current PID cut off frequency edit field
void __fastcall TForm3::PIDCurrentICutOffFrequencyEditKeyDown(TObject *Sender,
      WORD &Key, TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDCurrentICutOffFrequencyIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDCurrentICutOffFrequencyDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on current PID cut off frequency + button
void __fastcall TForm3::PIDCurrentICutOffFrequencyIncreaseButtonClick(
      TObject *Sender)
{
  IncreaseEditValue(PIDCurrentICutOffFrequencyEdit, 65535, 0, pIDCurrent.iCutOffFreq);
}
//---------------------------------------------------------------------------

//click on current PID cut off frequency - button
void __fastcall TForm3::PIDCurrentICutOffFrequencyDecreaseButtonClick(
      TObject *Sender)
{
  DecreaseEditValue(PIDCurrentICutOffFrequencyEdit, 65535, 0, pIDCurrent.iCutOffFreq);
}
//---------------------------------------------------------------------------

//write the PID setpoint setting to the fpga
//this is the setpoint used by both PID loops
void __fastcall TForm3::PIDPiezoSetPointEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(PIDPiezoSetPointEdit, 32767, -32768, pIDPiezo.setPoint);
  BYTE highBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_pIDPiezoSetPointRegister1,highBits);
  DoPutReg(reg_pIDPiezoSetPointRegister2,lowBits);
  DoneWriting();

  pIDPiezo.setPoint=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down key press on PID setpoint edit field
void __fastcall TForm3::PIDPiezoSetPointEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          PIDPiezoSetPointIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          PIDPiezoSetPointDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on PID setpoint gain + button
void __fastcall TForm3::PIDPiezoSetPointIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(PIDPiezoSetPointEdit, 32767, -32768, pIDPiezo.setPoint);
}
//---------------------------------------------------------------------------

//click on PID setpoint gain - button
void __fastcall TForm3::PIDPiezoSetPointDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(PIDPiezoSetPointEdit, 32767, -32768, pIDPiezo.setPoint);
}
//---------------------------------------------------------------------------

//write the scan setpoint bit to the FPGA
void __fastcall TForm3::ScanSetPointCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses2,oldBypasses);

  if (ScanSetPointCheckBox->Checked) {
    DoPutReg(reg_bypasses2, oldBypasses | 1<<reg_scanSetPointBit);

    scanSetPoint=true;
  }
  else {
    DoPutReg(reg_bypasses2, oldBypasses & ~(1<<reg_scanSetPointBit));

    scanSetPoint=false;
  }
}
//---------------------------------------------------------------------------


//write the input low pass enable bit to the FPGA
void __fastcall TForm3::InputLowPassEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (!InputLowPassEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_lowPass1BypassBit);

    lowPass1.bypass=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_lowPass1BypassBit));

    lowPass1.bypass=false;
  }
}
//---------------------------------------------------------------------------

//write the high pass enable bit to the FPGA
void __fastcall TForm3::InputHighPassEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (!InputHighPassEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_highPass1BypassBit);

    highPass1.bypass=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_highPass1BypassBit));

    highPass1.bypass=false;
  }
}
//---------------------------------------------------------------------------

//write the Scan enable bit to the FPGA
void __fastcall TForm3::ScanEnableCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (!ScanEnableCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_scanBypassBit);

    scan1.bypass=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_scanBypassBit));

    scan1.bypass=false;
  }
}
//---------------------------------------------------------------------------

//write the scan parameters to the fpga with a new scan amplitude
void __fastcall TForm3::ScanAmplitudeEditExit(TObject *Sender)
{
  scan1.amplitude=GetEditIntValue(ScanAmplitudeEdit, 65535, 0, scan1.amplitude);

  BYTE highBits=scan1.amplitude >> 8;
  BYTE lowBits=scan1.amplitude;
  DoPutReg(reg_scanAmplitudeRegister1,highBits);
  DoPutReg(reg_scanAmplitudeRegister2,lowBits);
  DoneWriting();

  ScanFrequencyEditExit(this);
}
//---------------------------------------------------------------------------

//cursor up/down press on scan amplitude buttons
void __fastcall TForm3::ScanAmplitudeEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          ScanAmplitudeIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          ScanAmplitudeDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on scan amplitude + button
void __fastcall TForm3::ScanAmplitudeIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(ScanAmplitudeEdit, 65535, 0, scan1.amplitude);
}
//---------------------------------------------------------------------------

//click on scan amplitude - button
void __fastcall TForm3::ScanAmplitudeDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(ScanAmplitudeEdit, 65535, 0, scan1.amplitude);
}
//---------------------------------------------------------------------------

//write the scan parameters to the fpga
void __fastcall TForm3::ScanFrequencyEditExit(TObject *Sender)
{
  scan1.frequency=GetEditIntValue(ScanFrequencyEdit, 16777215, 0, scan1.frequency);
  //sample clk on the fpga runs at 1 MHz, so the number of steps on the rising
  //edge at 1 Hz is 500000. The user entered parameter is in mHz
  int deltaX;
  int deltaY;
  if (scan1.amplitude==0 || scan1.frequency==0) {
    deltaY=0;
    deltaX=1;
    if (scan1.frequency==0) {
      //restart scan in case the scan frequency becomes 0 to force it to the
      //offset value regardless of amplitude
      ScanRestartButtonClick(this);
    }
  }
  else
  {
    int numStepsOnRisingEdge = 250000000/scan1.frequency;
    if (scan1.amplitude>numStepsOnRisingEdge) {
      deltaX=1;
      deltaY=int (double(scan1.amplitude)/double(numStepsOnRisingEdge));
    }
    else if (scan1.amplitude<=numStepsOnRisingEdge) {
      deltaY=1;
      deltaX=int (double(numStepsOnRisingEdge)/double(scan1.amplitude));
    }
  }

  BYTE highBits=deltaY >> 8;
  BYTE lowBits=deltaY;
  DoPutReg(reg_scanDeltaYRegister1,highBits);
  DoPutReg(reg_scanDeltaYRegister2,lowBits);
  DoneWriting();

  highBits=deltaX >> 8;
  lowBits=deltaX;
  DoPutReg(reg_scanDeltaXRegister1,highBits);
  DoPutReg(reg_scanDeltaXRegister2,lowBits);
  DoneWriting();
}
//---------------------------------------------------------------------------

//cursor up/down keypress in scan frequency edit field
void __fastcall TForm3::ScanFrequencyEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          ScanFrequencyIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          ScanFrequencyDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on scan frequency + button
void __fastcall TForm3::ScanFrequencyIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(ScanFrequencyEdit, 16777215, 0, scan1.frequency);
}
//---------------------------------------------------------------------------

//click on scan frequency - button
void __fastcall TForm3::ScanFrequencyDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(ScanFrequencyEdit, 16777215, 0, scan1.frequency);
}
//---------------------------------------------------------------------------

//write the scan offset value to the FPGA
void __fastcall TForm3::ScanOffsetEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(ScanOffsetEdit, 32768, -32767, scan1.offset);
  BYTE highBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_scanOffsetRegister1,highBits);
  DoPutReg(reg_scanOffsetRegister2,lowBits);
  DoneWriting();

  scan1.offset=newValue;
}
//---------------------------------------------------------------------------

//cursor up/down keypress in scan offset edit field
void __fastcall TForm3::ScanOffsetEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          ScanOffsetIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          ScanOffsetDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//click on scan offset + button
void __fastcall TForm3::ScanOffsetIncreaseButtonClick(TObject *Sender)
{
  IncreaseEditValue(ScanOffsetEdit, 32767, -32768, scan1.offset);
}
//---------------------------------------------------------------------------

//click on scan offset - button
void __fastcall TForm3::ScanOffsetDecreaseButtonClick(TObject *Sender)
{
  DecreaseEditValue(ScanOffsetEdit, 32767, -32768, scan1.offset);
}
//---------------------------------------------------------------------------

//restart the scan from zero. Useful for very slow scans.
//this function is called on scan restart button click,
void __fastcall TForm3::ScanRestartButtonClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses2,oldBypasses);

  DoPutReg(reg_bypasses2, oldBypasses | 1<<reg_scanRestartBit);
  Sleep(10);
  DoPutReg(reg_bypasses2, oldBypasses & ~(1<<reg_scanRestartBit));
}
//---------------------------------------------------------------------------

//write the input gain parameters to the FPGA
void __fastcall TForm3::InputGainEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(InputGainEdit, 65535, 0, input1.gain);
  BYTE highBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_inputGainRegister1,highBits);
  DoPutReg(reg_inputGainRegister2,lowBits);
  DoneWriting();

  input1.gain=newValue;
}
//---------------------------------------------------------------------------

//click on input gain + button
void __fastcall TForm3::InputGainIncreaseButtonClick(TObject *Sender)
{
    IncreaseEditValue(InputGainEdit, 65535, 0, input1.gain);
}
//---------------------------------------------------------------------------

//click input gain - button
void __fastcall TForm3::InputGainDecreaseButtonClick(TObject *Sender)
{
    DecreaseEditValue(InputGainEdit, 65535, 0, input1.gain);
}
//---------------------------------------------------------------------------

//cursor up/down keypress in input gain edit field
void __fastcall TForm3::InputGainEditKeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          InputGainIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          InputGainDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//write the input high pass cut off frequency to the FPGA
void __fastcall TForm3::InputHighPassCutOffFreqEditExit(TObject *Sender)
{
  int newValue=GetEditIntValue(InputHighPassCutOffFreqEdit, 16777215, 0, highPass1.cutOffFreq);

  BYTE highBits=newValue >> 16;
  BYTE mediumBits=newValue >> 8;
  BYTE lowBits=newValue;
  DoPutReg(reg_cutOffFreq2Register1,highBits);
  DoPutReg(reg_cutOffFreq2Register2,mediumBits);
  DoPutReg(reg_cutOffFreq2Register3,lowBits);
  DoneWriting();

  highPass1.cutOffFreq=newValue;
}
//---------------------------------------------------------------------------

//click on input high pass frequency + button
void __fastcall TForm3::InputHighPassCutOffFreqIncreaseButtonClick(
      TObject *Sender)
{
    IncreaseEditValue(InputHighPassCutOffFreqEdit, 16777215, 0, highPass1.cutOffFreq);
}
//---------------------------------------------------------------------------

//click on input high pass frequency - button
void __fastcall TForm3::InputHighPassCutOffFreqDecreaseButtonClick(
      TObject *Sender)
{
    DecreaseEditValue(InputHighPassCutOffFreqEdit, 16777215, 0, highPass1.cutOffFreq);

}
//---------------------------------------------------------------------------

//cursor up/down keypress in input high pass frequency edit field
void __fastcall TForm3::InputHighPassCutOffFreqEditKeyDown(TObject *Sender,
      WORD &Key, TShiftState Shift)
{
  switch (Key) {
    case VK_UP: {
          InputHighPassCutOffFreqIncreaseButtonClick(this);
          Key=0;
      };
      break;
    case VK_DOWN: {
          InputHighPassCutOffFreqDecreaseButtonClick(this);
          Key=0;
      };
      break;
    default:
      break;
  };
}
//---------------------------------------------------------------------------

//send the "locked" bit to the FPGA (turns on LOCKED mode)
void __fastcall TForm3::LockCheckBoxClick(TObject *Sender)
{
  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  if (LockCheckBox->Checked) {
    DoPutReg(reg_bypasses, oldBypasses | 1<<reg_lockedBit);

    lock=true;
  }
  else {
    DoPutReg(reg_bypasses,oldBypasses & ~(1<<reg_lockedBit));

    lock=false;
  }
}
//---------------------------------------------------------------------------

//"select device" menu selection, this allows to re-select the right usb port
void __fastcall TForm3::SelectDevice1Click(TObject *Sender)
{
  DoDvcTbl();
}
//---------------------------------------------------------------------------

//send the "bypass all" bit to the FPGA (this is a debugging mode,
//input=output when on
void __fastcall TForm3::BypassAllCheckBoxClick(TObject *Sender)
{
 BYTE oldBypasses=0;
  DoGetReg(reg_bypasses2,oldBypasses);

  if (BypassAllCheckBox->Checked) {
    DoPutReg(reg_bypasses2, oldBypasses | 1<<reg_bypassAllBit);
    bypassAll=true;
  }
  else {
    DoPutReg(reg_bypasses2,oldBypasses & ~(1<<reg_bypassAllBit));
    bypassAll=false;
  }
}
//---------------------------------------------------------------------------

//reset the program and FPGA on reset button click
void __fastcall TForm3::ResetButtonClick(TObject *Sender)
{
  FInit();
  UpdateGUIAndFPGA();

  BYTE oldBypasses=0;
  DoGetReg(reg_bypasses,oldBypasses);

  DoPutReg(reg_bypasses, oldBypasses | 1<<reg_resetBit);
  Sleep(10);
  DoPutReg(reg_bypasses, oldBypasses & ~(1<<reg_resetBit));
}
//---------------------------------------------------------------------------
